// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "content/browser/media/url_provision_fetcher.h"

#include "base/bind.h"
#include "base/metrics/histogram_functions.h"
#include "content/public/browser/provision_fetcher_factory.h"
#include "media/base/bind_to_current_loop.h"
#include "media/base/media_switches.h"
#include "net/base/load_flags.h"
#include "net/http/http_util.h"
#include "net/traffic_annotation/network_traffic_annotation.h"
#include "services/network/public/cpp/resource_request.h"
#include "services/network/public/cpp/shared_url_loader_factory.h"
#include "services/network/public/cpp/simple_url_loader.h"

namespace content {

// Implementation of URLProvisionFetcher.

URLProvisionFetcher::URLProvisionFetcher(
    scoped_refptr<network::SharedURLLoaderFactory> url_loader_factory)
    : url_loader_factory_(std::move(url_loader_factory)) {
  DCHECK(url_loader_factory_);
}

URLProvisionFetcher::~URLProvisionFetcher() {}

void URLProvisionFetcher::Retrieve(
    const std::string& default_url,
    const std::string& request_data,
    media::ProvisionFetcher::ResponseCB response_cb) {
  // For testing, don't actually do provisioning if the feature is enabled,
  // just indicate that the request failed.
  if (base::FeatureList::IsEnabled(media::kFailUrlProvisionFetcherForTesting)) {
    std::move(response_cb).Run(false, std::string());
    return;
  }

  response_cb_ = std::move(response_cb);

  const std::string request_string =
      default_url + "&signedRequest=" + request_data;
  DVLOG(1) << __func__ << ": request:" << request_string;

  DCHECK(!simple_url_loader_);
  net::NetworkTrafficAnnotationTag traffic_annotation =
      net::DefineNetworkTrafficAnnotation("url_prevision_fetcher", R"(
        semantics {
          sender: "Content Decryption Module"
          description:
            "For a Content Decryption Module (CDM) to obtain origin-specific "
            "identifiers from an individualization or provisioning server. See "
            "https://w3c.github.io/encrypted-media/#direct-individualization."
          trigger:
            "During protected content playback, if the CDM hasn’t been "
            "provisioned yet, it may trigger a provision request which will be "
            "sent to a provisioning server."
          data:
            "Opaque provision request generated by the CDM. It may contain "
            "distinctive identifiers (see "
            "https://w3c.github.io/encrypted-media/#distinctive-identifier) "
            "and/or distinctive permanent identifiers (see "
            "https://w3c.github.io/encrypted-media/#distinctive-permanent-"
            "identifier), which must be encrypted. It does NOT contain origin "
            "information, even in encrypted form."
          destination: OTHER
        }
        policy {
          cookies_allowed: NO
          setting:
            "On Android, users can disable this feature by disabling Protected "
            "Media Identifier permissions."
          policy_exception_justification: "Not implemented."
        })");
  auto resource_request = std::make_unique<network::ResourceRequest>();
  resource_request->url = GURL(request_string);
  resource_request->credentials_mode = network::mojom::CredentialsMode::kOmit;
  resource_request->method = "POST";
  resource_request->headers.SetHeader("User-Agent", "Widevine CDM v1.0");
  simple_url_loader_ = network::SimpleURLLoader::Create(
      std::move(resource_request), traffic_annotation);
  simple_url_loader_->AttachStringForUpload("", "application/json");
  simple_url_loader_->DownloadToStringOfUnboundedSizeUntilCrashAndDie(
      url_loader_factory_.get(),
      base::BindOnce(&URLProvisionFetcher::OnSimpleLoaderComplete,
                     base::Unretained(this)));
}

void URLProvisionFetcher::OnSimpleLoaderComplete(
    std::unique_ptr<std::string> response_body) {
  bool success = false;
  int response_code = simple_url_loader_->NetError();
  std::string response;
  const auto& headers = simple_url_loader_->ResponseInfo()
                            ? simple_url_loader_->ResponseInfo()->headers
                            : nullptr;
  if (headers) {
    // If there is a valid header, use the HTTP response code instead of the
    // net::Error status.
    response_code =
        net::HttpUtil::MapStatusCodeForHistogram(headers->response_code());
  }

  if (response_body) {
    success = true;
    response = std::move(*response_body);
  } else {
    DVLOG(1) << "CDM provision: server returned error code " << response_code;
  }

  simple_url_loader_.reset();
  base::UmaHistogramSparse("Media.EME.UrlProvisionFetcher.ResponseCode",
                           response_code);
  std::move(response_cb_).Run(success, response);
}

// Implementation of content public method CreateProvisionFetcher().

std::unique_ptr<media::ProvisionFetcher> CreateProvisionFetcher(
    scoped_refptr<network::SharedURLLoaderFactory> url_loader_factory) {
  DCHECK(url_loader_factory);
  return std::make_unique<URLProvisionFetcher>(std::move(url_loader_factory));
}

}  // namespace content
