// SPDX-License-Identifier: GPL-2.0-or-later
/*
 * Copyright (C) 2016 IBM Corp.
 */
#include <linux/bitops.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/kernel.h>
#include <linux/mfd/syscon.h>
#include <linux/mutex.h>
#include <linux/of.h>
#include <linux/platform_device.h>
#include <linux/pinctrl/pinctrl.h>
#include <linux/pinctrl/pinmux.h>
#include <linux/pinctrl/pinconf.h>
#include <linux/pinctrl/pinconf-generic.h>
#include <linux/string.h>
#include <linux/types.h>

#include "../core.h"
#include "../pinctrl-utils.h"
#include "pinctrl-aspeed.h"

/* Wrap some of the common macros for clarity */
#define SIG_EXPR_DECL_SINGLE(sig, func, ...) \
	SIG_EXPR_DECL(sig, func, func, __VA_ARGS__)

#define SIG_EXPR_LIST_DECL_SINGLE SIG_EXPR_LIST_DECL_SESG
#define SIG_EXPR_LIST_DECL_DUAL SIG_EXPR_LIST_DECL_DESG

/*
 * The "Multi-function Pins Mapping and Control" table in the SoC datasheet
 * references registers by the device/offset mnemonic. The register macros
 * below are named the same way to ease transcription and verification (as
 * opposed to naming them e.g. PINMUX_CTRL_[0-9]). Further, signal expressions
 * reference registers beyond those dedicated to pinmux, such as the system
 * reset control and MAC clock configuration registers. The AST2500 goes a step
 * further and references registers in the graphics IP block.
 */
#define SCU2C           0x2C /* Misc. Control Register */
#define SCU3C           0x3C /* System Reset Control/Status Register */
#define SCU48           0x48 /* MAC Interface Clock Delay Setting */
#define HW_STRAP1       0x70 /* AST2400 strapping is 33 bits, is split */
#define HW_REVISION_ID  0x7C /* Silicon revision ID register */
#define SCU80           0x80 /* Multi-function Pin Control #1 */
#define SCU84           0x84 /* Multi-function Pin Control #2 */
#define SCU88           0x88 /* Multi-function Pin Control #3 */
#define SCU8C           0x8C /* Multi-function Pin Control #4 */
#define SCU90           0x90 /* Multi-function Pin Control #5 */
#define SCU94           0x94 /* Multi-function Pin Control #6 */
#define SCUA0           0xA0 /* Multi-function Pin Control #7 */
#define SCUA4           0xA4 /* Multi-function Pin Control #8 */
#define SCUA8           0xA8 /* Multi-function Pin Control #9 */
#define SCUAC           0xAC /* Multi-function Pin Control #10 */
#define HW_STRAP2       0xD0 /* Strapping */

#define ASPEED_G5_NR_PINS 236

#define COND1		{ ASPEED_IP_SCU, SCU90, BIT(6), 0, 0 }
#define COND2		{ ASPEED_IP_SCU, SCU94, GENMASK(1, 0), 0, 0 }

/* LHCR0 is offset from the end of the H8S/2168-compatible registers */
#define LHCR0		0xa0
#define GFX064		0x64

#define B14 0
SSSF_PIN_DECL(B14, GPIOA0, MAC1LINK, SIG_DESC_SET(SCU80, 0));

#define D14 1
SSSF_PIN_DECL(D14, GPIOA1, MAC2LINK, SIG_DESC_SET(SCU80, 1));

#define D13 2
SIG_EXPR_LIST_DECL_SINGLE(D13, SPI1CS1, SPI1CS1, SIG_DESC_SET(SCU80, 15));
SIG_EXPR_LIST_DECL_SINGLE(D13, TIMER3, TIMER3, SIG_DESC_SET(SCU80, 2));
PIN_DECL_2(D13, GPIOA2, SPI1CS1, TIMER3);
FUNC_GROUP_DECL(SPI1CS1, D13);
FUNC_GROUP_DECL(TIMER3, D13);

#define E13 3
SSSF_PIN_DECL(E13, GPIOA3, TIMER4, SIG_DESC_SET(SCU80, 3));

#define I2C9_DESC	SIG_DESC_SET(SCU90, 22)

#define C14 4
SIG_EXPR_LIST_DECL_SINGLE(C14, SCL9, I2C9, I2C9_DESC, COND1);
SIG_EXPR_LIST_DECL_SINGLE(C14, TIMER5, TIMER5, SIG_DESC_SET(SCU80, 4), COND1);
PIN_DECL_2(C14, GPIOA4, SCL9, TIMER5);

FUNC_GROUP_DECL(TIMER5, C14);

#define A13 5
SIG_EXPR_LIST_DECL_SINGLE(A13, SDA9, I2C9, I2C9_DESC, COND1);
SIG_EXPR_LIST_DECL_SINGLE(A13, TIMER6, TIMER6, SIG_DESC_SET(SCU80, 5), COND1);
PIN_DECL_2(A13, GPIOA5, SDA9, TIMER6);

FUNC_GROUP_DECL(TIMER6, A13);

FUNC_GROUP_DECL(I2C9, C14, A13);

#define MDIO2_DESC	SIG_DESC_SET(SCU90, 2)

#define C13 6
SIG_EXPR_LIST_DECL_SINGLE(C13, MDC2, MDIO2, MDIO2_DESC, COND1);
SIG_EXPR_LIST_DECL_SINGLE(C13, TIMER7, TIMER7, SIG_DESC_SET(SCU80, 6), COND1);
PIN_DECL_2(C13, GPIOA6, MDC2, TIMER7);

FUNC_GROUP_DECL(TIMER7, C13);

#define B13 7
SIG_EXPR_LIST_DECL_SINGLE(B13, MDIO2, MDIO2, MDIO2_DESC, COND1);
SIG_EXPR_LIST_DECL_SINGLE(B13, TIMER8, TIMER8, SIG_DESC_SET(SCU80, 7), COND1);
PIN_DECL_2(B13, GPIOA7, MDIO2, TIMER8);

FUNC_GROUP_DECL(TIMER8, B13);

FUNC_GROUP_DECL(MDIO2, C13, B13);

#define K19 8
GPIO_PIN_DECL(K19, GPIOB0);

#define L19 9
GPIO_PIN_DECL(L19, GPIOB1);

#define L18 10
GPIO_PIN_DECL(L18, GPIOB2);

#define K18 11
GPIO_PIN_DECL(K18, GPIOB3);

#define J20 12
SSSF_PIN_DECL(J20, GPIOB4, USBCKI, SIG_DESC_SET(HW_STRAP1, 23));

#define H21 13
#define H21_DESC	SIG_DESC_SET(SCU80, 13)
SIG_EXPR_LIST_DECL_SINGLE(H21, LPCPD, LPCPD, H21_DESC);
SIG_EXPR_LIST_DECL_SINGLE(H21, LPCSMI, LPCSMI, H21_DESC);
PIN_DECL_2(H21, GPIOB5, LPCPD, LPCSMI);
FUNC_GROUP_DECL(LPCPD, H21);
FUNC_GROUP_DECL(LPCSMI, H21);

#define H22 14
SSSF_PIN_DECL(H22, GPIOB6, LPCPME, SIG_DESC_SET(SCU80, 14));

#define H20 15
GPIO_PIN_DECL(H20, GPIOB7);

#define SD1_DESC	SIG_DESC_SET(SCU90, 0)

#define C12 16
#define I2C10_DESC	SIG_DESC_SET(SCU90, 23)
SIG_EXPR_LIST_DECL_SINGLE(C12, SD1CLK, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(C12, SCL10, I2C10, I2C10_DESC);
PIN_DECL_2(C12, GPIOC0, SD1CLK, SCL10);

#define A12 17
SIG_EXPR_LIST_DECL_SINGLE(A12, SD1CMD, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(A12, SDA10, I2C10, I2C10_DESC);
PIN_DECL_2(A12, GPIOC1, SD1CMD, SDA10);

FUNC_GROUP_DECL(I2C10, C12, A12);

#define B12 18
#define I2C11_DESC	SIG_DESC_SET(SCU90, 24)
SIG_EXPR_LIST_DECL_SINGLE(B12, SD1DAT0, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(B12, SCL11, I2C11, I2C11_DESC);
PIN_DECL_2(B12, GPIOC2, SD1DAT0, SCL11);

#define D9  19
SIG_EXPR_LIST_DECL_SINGLE(D9, SD1DAT1, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(D9, SDA11, I2C11, I2C11_DESC);
PIN_DECL_2(D9, GPIOC3, SD1DAT1, SDA11);

FUNC_GROUP_DECL(I2C11, B12, D9);

#define D10 20
#define I2C12_DESC	SIG_DESC_SET(SCU90, 25)
SIG_EXPR_LIST_DECL_SINGLE(D10, SD1DAT2, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(D10, SCL12, I2C12, I2C12_DESC);
PIN_DECL_2(D10, GPIOC4, SD1DAT2, SCL12);

#define E12 21
SIG_EXPR_LIST_DECL_SINGLE(E12, SD1DAT3, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(E12, SDA12, I2C12, I2C12_DESC);
PIN_DECL_2(E12, GPIOC5, SD1DAT3, SDA12);

FUNC_GROUP_DECL(I2C12, D10, E12);

#define C11 22
#define I2C13_DESC	SIG_DESC_SET(SCU90, 26)
SIG_EXPR_LIST_DECL_SINGLE(C11, SD1CD, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(C11, SCL13, I2C13, I2C13_DESC);
PIN_DECL_2(C11, GPIOC6, SD1CD, SCL13);

#define B11 23
SIG_EXPR_LIST_DECL_SINGLE(B11, SD1WP, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(B11, SDA13, I2C13, I2C13_DESC);
PIN_DECL_2(B11, GPIOC7, SD1WP, SDA13);

FUNC_GROUP_DECL(I2C13, C11, B11);
FUNC_GROUP_DECL(SD1, C12, A12, B12, D9, D10, E12, C11, B11);

#define SD2_DESC        SIG_DESC_SET(SCU90, 1)
#define GPID0_DESC      SIG_DESC_SET(SCU8C, 8)
#define GPID_DESC       SIG_DESC_SET(HW_STRAP1, 21)

#define F19 24
SIG_EXPR_LIST_DECL_SINGLE(F19, SD2CLK, SD2, SD2_DESC);
SIG_EXPR_DECL_SINGLE(GPID0IN, GPID0, GPID0_DESC);
SIG_EXPR_DECL_SINGLE(GPID0IN, GPID, GPID_DESC);
SIG_EXPR_LIST_DECL_DUAL(F19, GPID0IN, GPID0, GPID);
PIN_DECL_2(F19, GPIOD0, SD2CLK, GPID0IN);

#define E21 25
SIG_EXPR_LIST_DECL_SINGLE(E21, SD2CMD, SD2, SD2_DESC);
SIG_EXPR_DECL_SINGLE(GPID0OUT, GPID0, GPID0_DESC);
SIG_EXPR_DECL_SINGLE(GPID0OUT, GPID, GPID_DESC);
SIG_EXPR_LIST_DECL_DUAL(E21, GPID0OUT, GPID0, GPID);
PIN_DECL_2(E21, GPIOD1, SD2CMD, GPID0OUT);

FUNC_GROUP_DECL(GPID0, F19, E21);

#define GPID2_DESC      SIG_DESC_SET(SCU8C, 9)

#define F20 26
SIG_EXPR_LIST_DECL_SINGLE(F20, SD2DAT0, SD2, SD2_DESC);
SIG_EXPR_DECL_SINGLE(GPID2IN, GPID2, GPID2_DESC);
SIG_EXPR_DECL_SINGLE(GPID2IN, GPID, GPID_DESC);
SIG_EXPR_LIST_DECL_DUAL(F20, GPID2IN, GPID2, GPID);
PIN_DECL_2(F20, GPIOD2, SD2DAT0, GPID2IN);

#define D20 27
SIG_EXPR_LIST_DECL_SINGLE(D20, SD2DAT1, SD2, SD2_DESC);
SIG_EXPR_DECL_SINGLE(GPID2OUT, GPID2, GPID2_DESC);
SIG_EXPR_DECL_SINGLE(GPID2OUT, GPID, GPID_DESC);
SIG_EXPR_LIST_DECL_DUAL(D20, GPID2OUT, GPID2, GPID);
PIN_DECL_2(D20, GPIOD3, SD2DAT1, GPID2OUT);

FUNC_GROUP_DECL(GPID2, F20, D20);

#define GPID4_DESC      SIG_DESC_SET(SCU8C, 10)

#define D21 28
SIG_EXPR_LIST_DECL_SINGLE(D21, SD2DAT2, SD2, SD2_DESC);
SIG_EXPR_DECL_SINGLE(GPID4IN, GPID4, GPID4_DESC);
SIG_EXPR_DECL_SINGLE(GPID4IN, GPID, GPID_DESC);
SIG_EXPR_LIST_DECL_DUAL(D21, GPID4IN, GPID4, GPID);
PIN_DECL_2(D21, GPIOD4, SD2DAT2, GPID4IN);

#define E20 29
SIG_EXPR_LIST_DECL_SINGLE(E20, SD2DAT3, SD2, SD2_DESC);
SIG_EXPR_DECL_SINGLE(GPID4OUT, GPID4, GPID4_DESC);
SIG_EXPR_DECL_SINGLE(GPID4OUT, GPID, GPID_DESC);
SIG_EXPR_LIST_DECL_DUAL(E20, GPID4OUT, GPID4, GPID);
PIN_DECL_2(E20, GPIOD5, SD2DAT3, GPID4OUT);

FUNC_GROUP_DECL(GPID4, D21, E20);

#define GPID6_DESC      SIG_DESC_SET(SCU8C, 11)

#define G18 30
SIG_EXPR_LIST_DECL_SINGLE(G18, SD2CD, SD2, SD2_DESC);
SIG_EXPR_DECL_SINGLE(GPID6IN, GPID6, GPID6_DESC);
SIG_EXPR_DECL_SINGLE(GPID6IN, GPID, GPID_DESC);
SIG_EXPR_LIST_DECL_DUAL(G18, GPID6IN, GPID6, GPID);
PIN_DECL_2(G18, GPIOD6, SD2CD, GPID6IN);

#define C21 31
SIG_EXPR_LIST_DECL_SINGLE(C21, SD2WP, SD2, SD2_DESC);
SIG_EXPR_DECL_SINGLE(GPID6OUT, GPID6, GPID6_DESC);
SIG_EXPR_DECL_SINGLE(GPID6OUT, GPID, GPID_DESC);
SIG_EXPR_LIST_DECL_DUAL(C21, GPID6OUT, GPID6, GPID);
PIN_DECL_2(C21, GPIOD7, SD2WP, GPID6OUT);

FUNC_GROUP_DECL(GPID6, G18, C21);
FUNC_GROUP_DECL(SD2, F19, E21, F20, D20, D21, E20, G18, C21);

#define GPIE_DESC	SIG_DESC_SET(HW_STRAP1, 22)
#define GPIE0_DESC	SIG_DESC_SET(SCU8C, 12)

#define B20 32
SIG_EXPR_LIST_DECL_SINGLE(B20, NCTS3, NCTS3, SIG_DESC_SET(SCU80, 16));
SIG_EXPR_DECL_SINGLE(GPIE0IN, GPIE0, GPIE0_DESC);
SIG_EXPR_DECL_SINGLE(GPIE0IN, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(B20, GPIE0IN, GPIE0, GPIE);
PIN_DECL_2(B20, GPIOE0, NCTS3, GPIE0IN);
FUNC_GROUP_DECL(NCTS3, B20);

#define C20 33
SIG_EXPR_LIST_DECL_SINGLE(C20, NDCD3, NDCD3, SIG_DESC_SET(SCU80, 17));
SIG_EXPR_DECL_SINGLE(GPIE0OUT, GPIE0, GPIE0_DESC);
SIG_EXPR_DECL_SINGLE(GPIE0OUT, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(C20, GPIE0OUT, GPIE0, GPIE);
PIN_DECL_2(C20, GPIOE1, NDCD3, GPIE0OUT);
FUNC_GROUP_DECL(NDCD3, C20);

FUNC_GROUP_DECL(GPIE0, B20, C20);

#define GPIE2_DESC	SIG_DESC_SET(SCU8C, 13)

#define F18 34
SIG_EXPR_LIST_DECL_SINGLE(F18, NDSR3, NDSR3, SIG_DESC_SET(SCU80, 18));
SIG_EXPR_DECL_SINGLE(GPIE2IN, GPIE2, GPIE2_DESC);
SIG_EXPR_DECL_SINGLE(GPIE2IN, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(F18, GPIE2IN, GPIE2, GPIE);
PIN_DECL_2(F18, GPIOE2, NDSR3, GPIE2IN);
FUNC_GROUP_DECL(NDSR3, F18);


#define F17 35
SIG_EXPR_LIST_DECL_SINGLE(F17, NRI3, NRI3, SIG_DESC_SET(SCU80, 19));
SIG_EXPR_DECL_SINGLE(GPIE2OUT, GPIE2, GPIE2_DESC);
SIG_EXPR_DECL_SINGLE(GPIE2OUT, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(F17, GPIE2OUT, GPIE2, GPIE);
PIN_DECL_2(F17, GPIOE3, NRI3, GPIE2OUT);
FUNC_GROUP_DECL(NRI3, F17);

FUNC_GROUP_DECL(GPIE2, F18, F17);

#define GPIE4_DESC	SIG_DESC_SET(SCU8C, 14)

#define E18 36
SIG_EXPR_LIST_DECL_SINGLE(E18, NDTR3, NDTR3, SIG_DESC_SET(SCU80, 20));
SIG_EXPR_DECL_SINGLE(GPIE4IN, GPIE4, GPIE4_DESC);
SIG_EXPR_DECL_SINGLE(GPIE4IN, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(E18, GPIE4IN, GPIE4, GPIE);
PIN_DECL_2(E18, GPIOE4, NDTR3, GPIE4IN);
FUNC_GROUP_DECL(NDTR3, E18);

#define D19 37
SIG_EXPR_LIST_DECL_SINGLE(D19, NRTS3, NRTS3, SIG_DESC_SET(SCU80, 21));
SIG_EXPR_DECL_SINGLE(GPIE4OUT, GPIE4, GPIE4_DESC);
SIG_EXPR_DECL_SINGLE(GPIE4OUT, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(D19, GPIE4OUT, GPIE4, GPIE);
PIN_DECL_2(D19, GPIOE5, NRTS3, GPIE4OUT);
FUNC_GROUP_DECL(NRTS3, D19);

FUNC_GROUP_DECL(GPIE4, E18, D19);

#define GPIE6_DESC	SIG_DESC_SET(SCU8C, 15)

#define A20 38
SIG_EXPR_LIST_DECL_SINGLE(A20, TXD3, TXD3, SIG_DESC_SET(SCU80, 22));
SIG_EXPR_DECL_SINGLE(GPIE6IN, GPIE6, GPIE6_DESC);
SIG_EXPR_DECL_SINGLE(GPIE6IN, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(A20, GPIE6IN, GPIE6, GPIE);
PIN_DECL_2(A20, GPIOE6, TXD3, GPIE6IN);
FUNC_GROUP_DECL(TXD3, A20);

#define B19 39
SIG_EXPR_LIST_DECL_SINGLE(B19, RXD3, RXD3, SIG_DESC_SET(SCU80, 23));
SIG_EXPR_DECL_SINGLE(GPIE6OUT, GPIE6, GPIE6_DESC);
SIG_EXPR_DECL_SINGLE(GPIE6OUT, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(B19, GPIE6OUT, GPIE6, GPIE);
PIN_DECL_2(B19, GPIOE7, RXD3, GPIE6OUT);
FUNC_GROUP_DECL(RXD3, B19);

FUNC_GROUP_DECL(GPIE6, A20, B19);

#define LPCHC_DESC	SIG_DESC_IP_SET(ASPEED_IP_LPC, LHCR0, 0)
#define LPCPLUS_DESC	SIG_DESC_SET(SCU90, 30)

#define J19 40
SIG_EXPR_DECL_SINGLE(LHAD0, LPCHC, LPCHC_DESC);
SIG_EXPR_DECL_SINGLE(LHAD0, LPCPLUS, LPCPLUS_DESC);
SIG_EXPR_LIST_DECL_DUAL(J19, LHAD0, LPCHC, LPCPLUS);
SIG_EXPR_LIST_DECL_SINGLE(J19, NCTS4, NCTS4, SIG_DESC_SET(SCU80, 24));
PIN_DECL_2(J19, GPIOF0, LHAD0, NCTS4);
FUNC_GROUP_DECL(NCTS4, J19);

#define J18 41
SIG_EXPR_DECL_SINGLE(LHAD1, LPCHC, LPCHC_DESC);
SIG_EXPR_DECL_SINGLE(LHAD1, LPCPLUS, LPCPLUS_DESC);
SIG_EXPR_LIST_DECL_DUAL(J18, LHAD1, LPCHC, LPCPLUS);
SIG_EXPR_LIST_DECL_SINGLE(J18, NDCD4, NDCD4, SIG_DESC_SET(SCU80, 25));
PIN_DECL_2(J18, GPIOF1, LHAD1, NDCD4);
FUNC_GROUP_DECL(NDCD4, J18);

#define B22 42
SIG_EXPR_DECL_SINGLE(LHAD2, LPCHC, LPCHC_DESC);
SIG_EXPR_DECL_SINGLE(LHAD2, LPCPLUS, LPCPLUS_DESC);
SIG_EXPR_LIST_DECL_DUAL(B22, LHAD2, LPCHC, LPCPLUS);
SIG_EXPR_LIST_DECL_SINGLE(B22, NDSR4, NDSR4, SIG_DESC_SET(SCU80, 26));
PIN_DECL_2(B22, GPIOF2, LHAD2, NDSR4);
FUNC_GROUP_DECL(NDSR4, B22);

#define B21 43
SIG_EXPR_DECL_SINGLE(LHAD3, LPCHC, LPCHC_DESC);
SIG_EXPR_DECL_SINGLE(LHAD3, LPCPLUS, LPCPLUS_DESC);
SIG_EXPR_LIST_DECL_DUAL(B21, LHAD3, LPCHC, LPCPLUS);
SIG_EXPR_LIST_DECL_SINGLE(B21, NRI4, NRI4, SIG_DESC_SET(SCU80, 27));
PIN_DECL_2(B21, GPIOF3, LHAD3, NRI4);
FUNC_GROUP_DECL(NRI4, B21);

#define A21 44
SIG_EXPR_DECL_SINGLE(LHCLK, LPCHC, LPCHC_DESC);
SIG_EXPR_DECL_SINGLE(LHCLK, LPCPLUS, LPCPLUS_DESC);
SIG_EXPR_LIST_DECL_DUAL(A21, LHCLK, LPCHC, LPCPLUS);
SIG_EXPR_LIST_DECL_SINGLE(A21, NDTR4, NDTR4, SIG_DESC_SET(SCU80, 28));
PIN_DECL_2(A21, GPIOF4, LHCLK, NDTR4);
FUNC_GROUP_DECL(NDTR4, A21);

#define H19 45
SIG_EXPR_DECL_SINGLE(LHFRAME, LPCHC, LPCHC_DESC);
SIG_EXPR_DECL_SINGLE(LHFRAME, LPCPLUS, LPCPLUS_DESC);
SIG_EXPR_LIST_DECL_DUAL(H19, LHFRAME, LPCHC, LPCPLUS);
SIG_EXPR_LIST_DECL_SINGLE(H19, NRTS4, NRTS4, SIG_DESC_SET(SCU80, 29));
PIN_DECL_2(H19, GPIOF5, LHFRAME, NRTS4);
FUNC_GROUP_DECL(NRTS4, H19);

#define G17 46
SIG_EXPR_LIST_DECL_SINGLE(G17, LHSIRQ, LPCHC, LPCHC_DESC);
SIG_EXPR_LIST_DECL_SINGLE(G17, TXD4, TXD4, SIG_DESC_SET(SCU80, 30));
PIN_DECL_2(G17, GPIOF6, LHSIRQ, TXD4);
FUNC_GROUP_DECL(TXD4, G17);

#define H18 47
SIG_EXPR_DECL_SINGLE(LHRST, LPCHC, LPCHC_DESC);
SIG_EXPR_DECL_SINGLE(LHRST, LPCPLUS, LPCPLUS_DESC);
SIG_EXPR_LIST_DECL_DUAL(H18, LHRST, LPCHC, LPCPLUS);
SIG_EXPR_LIST_DECL_SINGLE(H18, RXD4, RXD4, SIG_DESC_SET(SCU80, 31));
PIN_DECL_2(H18, GPIOF7, LHRST, RXD4);
FUNC_GROUP_DECL(RXD4, H18);

FUNC_GROUP_DECL(LPCHC, J19, J18, B22, B21, A21, H19, G17, H18);
FUNC_GROUP_DECL(LPCPLUS, J19, J18, B22, B21, A21, H19, H18);

#define A19 48
SIG_EXPR_LIST_DECL_SINGLE(A19, SGPS1CK, SGPS1, COND1, SIG_DESC_SET(SCU84, 0));
PIN_DECL_1(A19, GPIOG0, SGPS1CK);

#define E19 49
SIG_EXPR_LIST_DECL_SINGLE(E19, SGPS1LD, SGPS1, COND1, SIG_DESC_SET(SCU84, 1));
PIN_DECL_1(E19, GPIOG1, SGPS1LD);

#define C19 50
SIG_EXPR_LIST_DECL_SINGLE(C19, SGPS1I0, SGPS1, COND1, SIG_DESC_SET(SCU84, 2));
PIN_DECL_1(C19, GPIOG2, SGPS1I0);

#define E16 51
SIG_EXPR_LIST_DECL_SINGLE(E16, SGPS1I1, SGPS1, COND1, SIG_DESC_SET(SCU84, 3));
PIN_DECL_1(E16, GPIOG3, SGPS1I1);

FUNC_GROUP_DECL(SGPS1, A19, E19, C19, E16);

#define SGPS2_DESC	SIG_DESC_SET(SCU94, 12)

#define E17 52
SIG_EXPR_LIST_DECL_SINGLE(E17, SGPS2CK, SGPS2, COND1, SGPS2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(E17, SALT1, SALT1, COND1, SIG_DESC_SET(SCU84, 4));
PIN_DECL_2(E17, GPIOG4, SGPS2CK, SALT1);
FUNC_GROUP_DECL(SALT1, E17);

#define D16 53
SIG_EXPR_LIST_DECL_SINGLE(D16, SGPS2LD, SGPS2, COND1, SGPS2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(D16, SALT2, SALT2, COND1, SIG_DESC_SET(SCU84, 5));
PIN_DECL_2(D16, GPIOG5, SGPS2LD, SALT2);
FUNC_GROUP_DECL(SALT2, D16);

#define D15 54
SIG_EXPR_LIST_DECL_SINGLE(D15, SGPS2I0, SGPS2, COND1, SGPS2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(D15, SALT3, SALT3, COND1, SIG_DESC_SET(SCU84, 6));
PIN_DECL_2(D15, GPIOG6, SGPS2I0, SALT3);
FUNC_GROUP_DECL(SALT3, D15);

#define E14 55
SIG_EXPR_LIST_DECL_SINGLE(E14, SGPS2I1, SGPS2, COND1, SGPS2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(E14, SALT4, SALT4, COND1, SIG_DESC_SET(SCU84, 7));
PIN_DECL_2(E14, GPIOG7, SGPS2I1, SALT4);
FUNC_GROUP_DECL(SALT4, E14);

FUNC_GROUP_DECL(SGPS2, E17, D16, D15, E14);

#define UART6_DESC	SIG_DESC_SET(SCU90, 7)

#define A18 56
SIG_EXPR_LIST_DECL_SINGLE(A18, DASHA18, DASHA18, COND1, SIG_DESC_SET(SCU94, 5));
SIG_EXPR_LIST_DECL_SINGLE(A18, NCTS6, UART6, COND1, UART6_DESC);
PIN_DECL_2(A18, GPIOH0, DASHA18, NCTS6);

#define B18 57
SIG_EXPR_LIST_DECL_SINGLE(B18, DASHB18, DASHB18, COND1, SIG_DESC_SET(SCU94, 5));
SIG_EXPR_LIST_DECL_SINGLE(B18, NDCD6, UART6, COND1, UART6_DESC);
PIN_DECL_2(B18, GPIOH1, DASHB18, NDCD6);

#define D17 58
SIG_EXPR_LIST_DECL_SINGLE(D17, DASHD17, DASHD17, COND1, SIG_DESC_SET(SCU94, 6));
SIG_EXPR_LIST_DECL_SINGLE(D17, NDSR6, UART6, COND1, UART6_DESC);
PIN_DECL_2(D17, GPIOH2, DASHD17, NDSR6);

#define C17 59
SIG_EXPR_LIST_DECL_SINGLE(C17, DASHC17, DASHC17, COND1, SIG_DESC_SET(SCU94, 6));
SIG_EXPR_LIST_DECL_SINGLE(C17, NRI6, UART6, COND1, UART6_DESC);
PIN_DECL_2(C17, GPIOH3, DASHC17, NRI6);

#define A17 60
SIG_EXPR_LIST_DECL_SINGLE(A17, DASHA17, DASHA17, COND1, SIG_DESC_SET(SCU94, 7));
SIG_EXPR_LIST_DECL_SINGLE(A17, NDTR6, UART6, COND1, UART6_DESC);
PIN_DECL_2(A17, GPIOH4, DASHA17, NDTR6);

#define B17 61
SIG_EXPR_LIST_DECL_SINGLE(B17, DASHB17, DASHB17, COND1, SIG_DESC_SET(SCU94, 7));
SIG_EXPR_LIST_DECL_SINGLE(B17, NRTS6, UART6, COND1, UART6_DESC);
PIN_DECL_2(B17, GPIOH5, DASHB17, NRTS6);

#define A16 62
SIG_EXPR_LIST_DECL_SINGLE(A16, TXD6, UART6, COND1, UART6_DESC);
PIN_DECL_1(A16, GPIOH6, TXD6);

#define D18 63
SIG_EXPR_LIST_DECL_SINGLE(D18, RXD6, UART6, COND1, UART6_DESC);
PIN_DECL_1(D18, GPIOH7, RXD6);

FUNC_GROUP_DECL(UART6, A18, B18, D17, C17, A17, B17, A16, D18);

#define SPI1_DESC \
	{ ASPEED_IP_SCU, HW_STRAP1, GENMASK(13, 12), 1, 0 }
#define SPI1DEBUG_DESC \
	{ ASPEED_IP_SCU, HW_STRAP1, GENMASK(13, 12), 2, 0 }
#define SPI1PASSTHRU_DESC \
	{ ASPEED_IP_SCU, HW_STRAP1, GENMASK(13, 12), 3, 0 }

#define C18 64
SIG_EXPR_DECL_SINGLE(SYSCS, SPI1DEBUG, COND1, SPI1DEBUG_DESC);
SIG_EXPR_DECL_SINGLE(SYSCS, SPI1PASSTHRU, COND1, SPI1PASSTHRU_DESC);
SIG_EXPR_LIST_DECL_DUAL(C18, SYSCS, SPI1DEBUG, SPI1PASSTHRU);
PIN_DECL_1(C18, GPIOI0, SYSCS);

#define E15 65
SIG_EXPR_DECL_SINGLE(SYSCK, SPI1DEBUG, COND1, SPI1DEBUG_DESC);
SIG_EXPR_DECL_SINGLE(SYSCK, SPI1PASSTHRU, COND1, SPI1PASSTHRU_DESC);
SIG_EXPR_LIST_DECL_DUAL(E15, SYSCK, SPI1DEBUG, SPI1PASSTHRU);
PIN_DECL_1(E15, GPIOI1, SYSCK);

#define B16 66
SIG_EXPR_DECL_SINGLE(SYSMOSI, SPI1DEBUG, COND1, SPI1DEBUG_DESC);
SIG_EXPR_DECL_SINGLE(SYSMOSI, SPI1PASSTHRU, COND1, SPI1PASSTHRU_DESC);
SIG_EXPR_LIST_DECL_DUAL(B16, SYSMOSI, SPI1DEBUG, SPI1PASSTHRU);
PIN_DECL_1(B16, GPIOI2, SYSMOSI);

#define C16 67
SIG_EXPR_DECL_SINGLE(SYSMISO, SPI1DEBUG, COND1, SPI1DEBUG_DESC);
SIG_EXPR_DECL_SINGLE(SYSMISO, SPI1PASSTHRU, COND1, SPI1PASSTHRU_DESC);
SIG_EXPR_LIST_DECL_DUAL(C16, SYSMISO, SPI1DEBUG, SPI1PASSTHRU);
PIN_DECL_1(C16, GPIOI3, SYSMISO);

#define VB_DESC	SIG_DESC_SET(HW_STRAP1, 5)

#define B15 68
SIG_EXPR_DECL_SINGLE(SPI1CS0, SPI1, COND1, SPI1_DESC);
SIG_EXPR_DECL_SINGLE(SPI1CS0, SPI1DEBUG, COND1, SPI1DEBUG_DESC);
SIG_EXPR_DECL_SINGLE(SPI1CS0, SPI1PASSTHRU, COND1, SPI1PASSTHRU_DESC);
SIG_EXPR_LIST_DECL(SPI1CS0, SPI1,
			    SIG_EXPR_PTR(SPI1CS0, SPI1),
			    SIG_EXPR_PTR(SPI1CS0, SPI1DEBUG),
			    SIG_EXPR_PTR(SPI1CS0, SPI1PASSTHRU));
SIG_EXPR_LIST_ALIAS(B15, SPI1CS0, SPI1);
SIG_EXPR_LIST_DECL_SINGLE(B15, VBCS, VGABIOSROM, COND1, VB_DESC);
PIN_DECL_2(B15, GPIOI4, SPI1CS0, VBCS);

#define C15 69
SIG_EXPR_DECL_SINGLE(SPI1CK, SPI1, COND1, SPI1_DESC);
SIG_EXPR_DECL_SINGLE(SPI1CK, SPI1DEBUG, COND1, SPI1DEBUG_DESC);
SIG_EXPR_DECL_SINGLE(SPI1CK, SPI1PASSTHRU, COND1, SPI1PASSTHRU_DESC);
SIG_EXPR_LIST_DECL(SPI1CK, SPI1,
			    SIG_EXPR_PTR(SPI1CK, SPI1),
			    SIG_EXPR_PTR(SPI1CK, SPI1DEBUG),
			    SIG_EXPR_PTR(SPI1CK, SPI1PASSTHRU));
SIG_EXPR_LIST_ALIAS(C15, SPI1CK, SPI1);
SIG_EXPR_LIST_DECL_SINGLE(C15, VBCK, VGABIOSROM, COND1, VB_DESC);
PIN_DECL_2(C15, GPIOI5, SPI1CK, VBCK);

#define A14 70
SIG_EXPR_DECL_SINGLE(SPI1MOSI, SPI1, COND1, SPI1_DESC);
SIG_EXPR_DECL_SINGLE(SPI1MOSI, SPI1DEBUG, COND1, SPI1DEBUG_DESC);
SIG_EXPR_DECL_SINGLE(SPI1MOSI, SPI1PASSTHRU, COND1, SPI1PASSTHRU_DESC);
SIG_EXPR_LIST_DECL(SPI1MOSI, SPI1,
			    SIG_EXPR_PTR(SPI1MOSI, SPI1),
			    SIG_EXPR_PTR(SPI1MOSI, SPI1DEBUG),
			    SIG_EXPR_PTR(SPI1MOSI, SPI1PASSTHRU));
SIG_EXPR_LIST_ALIAS(A14, SPI1MOSI, SPI1);
SIG_EXPR_LIST_DECL_SINGLE(A14, VBMOSI, VGABIOSROM, COND1, VB_DESC);
PIN_DECL_2(A14, GPIOI6, SPI1MOSI, VBMOSI);

#define A15 71
SIG_EXPR_DECL_SINGLE(SPI1MISO, SPI1, COND1, SPI1_DESC);
SIG_EXPR_DECL_SINGLE(SPI1MISO, SPI1DEBUG, COND1, SPI1DEBUG_DESC);
SIG_EXPR_DECL_SINGLE(SPI1MISO, SPI1PASSTHRU, COND1, SPI1PASSTHRU_DESC);
SIG_EXPR_LIST_DECL(SPI1MISO, SPI1,
			    SIG_EXPR_PTR(SPI1MISO, SPI1),
			    SIG_EXPR_PTR(SPI1MISO, SPI1DEBUG),
			    SIG_EXPR_PTR(SPI1MISO, SPI1PASSTHRU));
SIG_EXPR_LIST_ALIAS(A15, SPI1MISO, SPI1);
SIG_EXPR_LIST_DECL_SINGLE(A15, VBMISO, VGABIOSROM, COND1, VB_DESC);
PIN_DECL_2(A15, GPIOI7, SPI1MISO, VBMISO);

FUNC_GROUP_DECL(SPI1, B15, C15, A14, A15);
FUNC_GROUP_DECL(SPI1DEBUG, C18, E15, B16, C16, B15, C15, A14, A15);
FUNC_GROUP_DECL(SPI1PASSTHRU, C18, E15, B16, C16, B15, C15, A14, A15);
FUNC_GROUP_DECL(VGABIOSROM, B15, C15, A14, A15);

#define R2 72
SIG_EXPR_LIST_DECL_SINGLE(R2, SGPMCK, SGPM, SIG_DESC_SET(SCU84, 8));
PIN_DECL_1(R2, GPIOJ0, SGPMCK);

#define L2 73
SIG_EXPR_LIST_DECL_SINGLE(L2, SGPMLD, SGPM, SIG_DESC_SET(SCU84, 9));
PIN_DECL_1(L2, GPIOJ1, SGPMLD);

#define N3 74
SIG_EXPR_LIST_DECL_SINGLE(N3, SGPMO, SGPM, SIG_DESC_SET(SCU84, 10));
PIN_DECL_1(N3, GPIOJ2, SGPMO);

#define N4 75
SIG_EXPR_LIST_DECL_SINGLE(N4, SGPMI, SGPM, SIG_DESC_SET(SCU84, 11));
PIN_DECL_1(N4, GPIOJ3, SGPMI);

FUNC_GROUP_DECL(SGPM, R2, L2, N3, N4);

#define N5 76
SIG_EXPR_LIST_DECL_SINGLE(N5, VGAHS, VGAHS, SIG_DESC_SET(SCU84, 12));
SIG_EXPR_LIST_DECL_SINGLE(N5, DASHN5, DASHN5, SIG_DESC_SET(SCU94, 8));
PIN_DECL_2(N5, GPIOJ4, VGAHS, DASHN5);
FUNC_GROUP_DECL(VGAHS, N5);

#define R4 77
SIG_EXPR_LIST_DECL_SINGLE(R4, VGAVS, VGAVS, SIG_DESC_SET(SCU84, 13));
SIG_EXPR_LIST_DECL_SINGLE(R4, DASHR4, DASHR4, SIG_DESC_SET(SCU94, 8));
PIN_DECL_2(R4, GPIOJ5, VGAVS, DASHR4);
FUNC_GROUP_DECL(VGAVS, R4);

#define R3 78
SIG_EXPR_LIST_DECL_SINGLE(R3, DDCCLK, DDCCLK, SIG_DESC_SET(SCU84, 14));
SIG_EXPR_LIST_DECL_SINGLE(R3, DASHR3, DASHR3, SIG_DESC_SET(SCU94, 9));
PIN_DECL_2(R3, GPIOJ6, DDCCLK, DASHR3);
FUNC_GROUP_DECL(DDCCLK, R3);

#define T3 79
SIG_EXPR_LIST_DECL_SINGLE(T3, DDCDAT, DDCDAT, SIG_DESC_SET(SCU84, 15));
SIG_EXPR_LIST_DECL_SINGLE(T3, DASHT3, DASHT3, SIG_DESC_SET(SCU94, 9));
PIN_DECL_2(T3, GPIOJ7, DDCDAT, DASHT3);
FUNC_GROUP_DECL(DDCDAT, T3);

#define I2C5_DESC       SIG_DESC_SET(SCU90, 18)

#define L3 80
SIG_EXPR_LIST_DECL_SINGLE(L3, SCL5, I2C5, I2C5_DESC);
PIN_DECL_1(L3, GPIOK0, SCL5);

#define L4 81
SIG_EXPR_LIST_DECL_SINGLE(L4, SDA5, I2C5, I2C5_DESC);
PIN_DECL_1(L4, GPIOK1, SDA5);

FUNC_GROUP_DECL(I2C5, L3, L4);

#define I2C6_DESC       SIG_DESC_SET(SCU90, 19)

#define L1 82
SIG_EXPR_LIST_DECL_SINGLE(L1, SCL6, I2C6, I2C6_DESC);
PIN_DECL_1(L1, GPIOK2, SCL6);

#define N2 83
SIG_EXPR_LIST_DECL_SINGLE(N2, SDA6, I2C6, I2C6_DESC);
PIN_DECL_1(N2, GPIOK3, SDA6);

FUNC_GROUP_DECL(I2C6, L1, N2);

#define I2C7_DESC       SIG_DESC_SET(SCU90, 20)

#define N1 84
SIG_EXPR_LIST_DECL_SINGLE(N1, SCL7, I2C7, I2C7_DESC);
PIN_DECL_1(N1, GPIOK4, SCL7);

#define P1 85
SIG_EXPR_LIST_DECL_SINGLE(P1, SDA7, I2C7, I2C7_DESC);
PIN_DECL_1(P1, GPIOK5, SDA7);

FUNC_GROUP_DECL(I2C7, N1, P1);

#define I2C8_DESC       SIG_DESC_SET(SCU90, 21)

#define P2 86
SIG_EXPR_LIST_DECL_SINGLE(P2, SCL8, I2C8, I2C8_DESC);
PIN_DECL_1(P2, GPIOK6, SCL8);

#define R1 87
SIG_EXPR_LIST_DECL_SINGLE(R1, SDA8, I2C8, I2C8_DESC);
PIN_DECL_1(R1, GPIOK7, SDA8);

FUNC_GROUP_DECL(I2C8, P2, R1);

#define T2 88
SSSF_PIN_DECL(T2, GPIOL0, NCTS1, SIG_DESC_SET(SCU84, 16));

#define VPIOFF0_DESC    { ASPEED_IP_SCU, SCU90, GENMASK(5, 4), 0, 0 }
#define VPIOFF1_DESC    { ASPEED_IP_SCU, SCU90, GENMASK(5, 4), 1, 0 }
#define VPI24_DESC      { ASPEED_IP_SCU, SCU90, GENMASK(5, 4), 2, 0 }
#define VPIRSVD_DESC    { ASPEED_IP_SCU, SCU90, GENMASK(5, 4), 3, 0 }
#define VPI_24_RSVD_DESC	SIG_DESC_SET(SCU90, 5)

#define T1 89
#define T1_DESC		SIG_DESC_SET(SCU84, 17)
SIG_EXPR_LIST_DECL_SINGLE(T1, VPIDE, VPI24, VPI_24_RSVD_DESC, T1_DESC, COND2);
SIG_EXPR_LIST_DECL_SINGLE(T1, NDCD1, NDCD1, T1_DESC, COND2);
PIN_DECL_2(T1, GPIOL1, VPIDE, NDCD1);
FUNC_GROUP_DECL(NDCD1, T1);

#define U1 90
#define U1_DESC		SIG_DESC_SET(SCU84, 18)
SIG_EXPR_LIST_DECL_SINGLE(U1, DASHU1, VPI24, VPI_24_RSVD_DESC, U1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(U1, NDSR1, NDSR1, U1_DESC);
PIN_DECL_2(U1, GPIOL2, DASHU1, NDSR1);
FUNC_GROUP_DECL(NDSR1, U1);

#define U2 91
#define U2_DESC		SIG_DESC_SET(SCU84, 19)
SIG_EXPR_LIST_DECL_SINGLE(U2, VPIHS, VPI24, VPI_24_RSVD_DESC, U2_DESC, COND2);
SIG_EXPR_LIST_DECL_SINGLE(U2, NRI1, NRI1, U2_DESC, COND2);
PIN_DECL_2(U2, GPIOL3, VPIHS, NRI1);
FUNC_GROUP_DECL(NRI1, U2);

#define P4 92
#define P4_DESC		SIG_DESC_SET(SCU84, 20)
SIG_EXPR_LIST_DECL_SINGLE(P4, VPIVS, VPI24, VPI_24_RSVD_DESC, P4_DESC, COND2);
SIG_EXPR_LIST_DECL_SINGLE(P4, NDTR1, NDTR1, P4_DESC, COND2);
PIN_DECL_2(P4, GPIOL4, VPIVS, NDTR1);
FUNC_GROUP_DECL(NDTR1, P4);

#define P3 93
#define P3_DESC		SIG_DESC_SET(SCU84, 21)
SIG_EXPR_LIST_DECL_SINGLE(P3, VPICLK, VPI24, VPI_24_RSVD_DESC, P3_DESC, COND2);
SIG_EXPR_LIST_DECL_SINGLE(P3, NRTS1, NRTS1, P3_DESC, COND2);
PIN_DECL_2(P3, GPIOL5, VPICLK, NRTS1);
FUNC_GROUP_DECL(NRTS1, P3);

#define V1 94
#define V1_DESC		SIG_DESC_SET(SCU84, 22)
SIG_EXPR_LIST_DECL_SINGLE(V1, DASHV1, DASHV1, VPIRSVD_DESC, V1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(V1, TXD1, TXD1, V1_DESC, COND2);
PIN_DECL_2(V1, GPIOL6, DASHV1, TXD1);
FUNC_GROUP_DECL(TXD1, V1);

#define W1 95
#define W1_DESC		SIG_DESC_SET(SCU84, 23)
SIG_EXPR_LIST_DECL_SINGLE(W1, DASHW1, DASHW1, VPIRSVD_DESC, W1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(W1, RXD1, RXD1, W1_DESC, COND2);
PIN_DECL_2(W1, GPIOL7, DASHW1, RXD1);
FUNC_GROUP_DECL(RXD1, W1);

#define Y1 96
#define Y1_DESC		SIG_DESC_SET(SCU84, 24)
SIG_EXPR_LIST_DECL_SINGLE(Y1, VPIB2, VPI24, VPI_24_RSVD_DESC, Y1_DESC, COND2);
SIG_EXPR_LIST_DECL_SINGLE(Y1, NCTS2, NCTS2, Y1_DESC, COND2);
PIN_DECL_2(Y1, GPIOM0, VPIB2, NCTS2);
FUNC_GROUP_DECL(NCTS2, Y1);

#define AB2 97
#define AB2_DESC	SIG_DESC_SET(SCU84, 25)
SIG_EXPR_LIST_DECL_SINGLE(AB2, VPIB3, VPI24, VPI_24_RSVD_DESC, AB2_DESC, COND2);
SIG_EXPR_LIST_DECL_SINGLE(AB2, NDCD2, NDCD2, AB2_DESC, COND2);
PIN_DECL_2(AB2, GPIOM1, VPIB3, NDCD2);
FUNC_GROUP_DECL(NDCD2, AB2);

#define AA1 98
#define AA1_DESC	SIG_DESC_SET(SCU84, 26)
SIG_EXPR_LIST_DECL_SINGLE(AA1, VPIB4, VPI24, VPI_24_RSVD_DESC, AA1_DESC, COND2);
SIG_EXPR_LIST_DECL_SINGLE(AA1, NDSR2, NDSR2, AA1_DESC, COND2);
PIN_DECL_2(AA1, GPIOM2, VPIB4, NDSR2);
FUNC_GROUP_DECL(NDSR2, AA1);

#define Y2 99
#define Y2_DESC		SIG_DESC_SET(SCU84, 27)
SIG_EXPR_LIST_DECL_SINGLE(Y2, VPIB5, VPI24, VPI_24_RSVD_DESC, Y2_DESC, COND2);
SIG_EXPR_LIST_DECL_SINGLE(Y2, NRI2, NRI2, Y2_DESC, COND2);
PIN_DECL_2(Y2, GPIOM3, VPIB5, NRI2);
FUNC_GROUP_DECL(NRI2, Y2);

#define AA2 100
#define AA2_DESC	SIG_DESC_SET(SCU84, 28)
SIG_EXPR_LIST_DECL_SINGLE(AA2, VPIB6, VPI24, VPI_24_RSVD_DESC, AA2_DESC, COND2);
SIG_EXPR_LIST_DECL_SINGLE(AA2, NDTR2, NDTR2, AA2_DESC, COND2);
PIN_DECL_2(AA2, GPIOM4, VPIB6, NDTR2);
FUNC_GROUP_DECL(NDTR2, AA2);

#define P5 101
#define P5_DESC	SIG_DESC_SET(SCU84, 29)
SIG_EXPR_LIST_DECL_SINGLE(P5, VPIB7, VPI24, VPI_24_RSVD_DESC, P5_DESC, COND2);
SIG_EXPR_LIST_DECL_SINGLE(P5, NRTS2, NRTS2, P5_DESC, COND2);
PIN_DECL_2(P5, GPIOM5, VPIB7, NRTS2);
FUNC_GROUP_DECL(NRTS2, P5);

#define R5 102
#define R5_DESC	SIG_DESC_SET(SCU84, 30)
SIG_EXPR_LIST_DECL_SINGLE(R5, VPIB8, VPI24, VPI_24_RSVD_DESC, R5_DESC, COND2);
SIG_EXPR_LIST_DECL_SINGLE(R5, TXD2, TXD2, R5_DESC, COND2);
PIN_DECL_2(R5, GPIOM6, VPIB8, TXD2);
FUNC_GROUP_DECL(TXD2, R5);

#define T5 103
#define T5_DESC	SIG_DESC_SET(SCU84, 31)
SIG_EXPR_LIST_DECL_SINGLE(T5, VPIB9, VPI24, VPI_24_RSVD_DESC, T5_DESC, COND2);
SIG_EXPR_LIST_DECL_SINGLE(T5, RXD2, RXD2, T5_DESC, COND2);
PIN_DECL_2(T5, GPIOM7, VPIB9, RXD2);
FUNC_GROUP_DECL(RXD2, T5);

#define V2 104
#define V2_DESC         SIG_DESC_SET(SCU88, 0)
SIG_EXPR_LIST_DECL_SINGLE(V2, DASHN0, DASHN0, VPIRSVD_DESC, V2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(V2, PWM0, PWM0, V2_DESC, COND2);
PIN_DECL_2(V2, GPION0, DASHN0, PWM0);
FUNC_GROUP_DECL(PWM0, V2);

#define W2 105
#define W2_DESC         SIG_DESC_SET(SCU88, 1)
SIG_EXPR_LIST_DECL_SINGLE(W2, DASHN1, DASHN1, VPIRSVD_DESC, W2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(W2, PWM1, PWM1, W2_DESC, COND2);
PIN_DECL_2(W2, GPION1, DASHN1, PWM1);
FUNC_GROUP_DECL(PWM1, W2);

#define V3 106
#define V3_DESC         SIG_DESC_SET(SCU88, 2)
SIG_EXPR_DECL_SINGLE(VPIG2, VPI24, VPI24_DESC, V3_DESC, COND2);
SIG_EXPR_DECL_SINGLE(VPIG2, VPIRSVD, VPIRSVD_DESC, V3_DESC, COND2);
SIG_EXPR_LIST_DECL_DUAL(V3, VPIG2, VPI24, VPIRSVD);
SIG_EXPR_LIST_DECL_SINGLE(V3, PWM2, PWM2, V3_DESC, COND2);
PIN_DECL_2(V3, GPION2, VPIG2, PWM2);
FUNC_GROUP_DECL(PWM2, V3);

#define U3 107
#define U3_DESC         SIG_DESC_SET(SCU88, 3)
SIG_EXPR_DECL_SINGLE(VPIG3, VPI24, VPI24_DESC, U3_DESC, COND2);
SIG_EXPR_DECL_SINGLE(VPIG3, VPIRSVD, VPIRSVD_DESC, U3_DESC, COND2);
SIG_EXPR_LIST_DECL_DUAL(U3, VPIG3, VPI24, VPIRSVD);
SIG_EXPR_LIST_DECL_SINGLE(U3, PWM3, PWM3, U3_DESC, COND2);
PIN_DECL_2(U3, GPION3, VPIG3, PWM3);
FUNC_GROUP_DECL(PWM3, U3);

#define W3 108
#define W3_DESC         SIG_DESC_SET(SCU88, 4)
SIG_EXPR_DECL_SINGLE(VPIG4, VPI24, VPI24_DESC, W3_DESC, COND2);
SIG_EXPR_DECL_SINGLE(VPIG4, VPIRSVD, VPIRSVD_DESC, W3_DESC, COND2);
SIG_EXPR_LIST_DECL_DUAL(W3, VPIG4, VPI24, VPIRSVD);
SIG_EXPR_LIST_DECL_SINGLE(W3, PWM4, PWM4, W3_DESC, COND2);
PIN_DECL_2(W3, GPION4, VPIG4, PWM4);
FUNC_GROUP_DECL(PWM4, W3);

#define AA3 109
#define AA3_DESC        SIG_DESC_SET(SCU88, 5)
SIG_EXPR_DECL_SINGLE(VPIG5, VPI24, VPI24_DESC, AA3_DESC, COND2);
SIG_EXPR_DECL_SINGLE(VPIG5, VPIRSVD, VPIRSVD_DESC, AA3_DESC, COND2);
SIG_EXPR_LIST_DECL_DUAL(AA3, VPIG5, VPI24, VPIRSVD);
SIG_EXPR_LIST_DECL_SINGLE(AA3, PWM5, PWM5, AA3_DESC, COND2);
PIN_DECL_2(AA3, GPION5, VPIG5, PWM5);
FUNC_GROUP_DECL(PWM5, AA3);

#define Y3 110
#define Y3_DESC         SIG_DESC_SET(SCU88, 6)
SIG_EXPR_LIST_DECL_SINGLE(Y3, VPIG6, VPI24, VPI24_DESC, Y3_DESC);
SIG_EXPR_LIST_DECL_SINGLE(Y3, PWM6, PWM6, Y3_DESC, COND2);
PIN_DECL_2(Y3, GPION6, VPIG6, PWM6);
FUNC_GROUP_DECL(PWM6, Y3);

#define T4 111
#define T4_DESC         SIG_DESC_SET(SCU88, 7)
SIG_EXPR_LIST_DECL_SINGLE(T4, VPIG7, VPI24, VPI24_DESC, T4_DESC);
SIG_EXPR_LIST_DECL_SINGLE(T4, PWM7, PWM7, T4_DESC, COND2);
PIN_DECL_2(T4, GPION7, VPIG7, PWM7);
FUNC_GROUP_DECL(PWM7, T4);

#define U5 112
SIG_EXPR_LIST_DECL_SINGLE(U5, VPIG8, VPI24, VPI24_DESC, SIG_DESC_SET(SCU88, 8),
			  COND2);
PIN_DECL_1(U5, GPIOO0, VPIG8);

#define U4 113
SIG_EXPR_LIST_DECL_SINGLE(U4, VPIG9, VPI24, VPI24_DESC, SIG_DESC_SET(SCU88, 9),
			  COND2);
PIN_DECL_1(U4, GPIOO1, VPIG9);

#define V5 114
SIG_EXPR_LIST_DECL_SINGLE(V5, DASHV5, DASHV5, VPI_24_RSVD_DESC,
			  SIG_DESC_SET(SCU88, 10));
PIN_DECL_1(V5, GPIOO2, DASHV5);

#define AB4 115
SIG_EXPR_LIST_DECL_SINGLE(AB4, DASHAB4, DASHAB4, VPI_24_RSVD_DESC,
			  SIG_DESC_SET(SCU88, 11));
PIN_DECL_1(AB4, GPIOO3, DASHAB4);

#define AB3 116
SIG_EXPR_LIST_DECL_SINGLE(AB3, VPIR2, VPI24, VPI24_DESC,
			  SIG_DESC_SET(SCU88, 12), COND2);
PIN_DECL_1(AB3, GPIOO4, VPIR2);

#define Y4 117
SIG_EXPR_LIST_DECL_SINGLE(Y4, VPIR3, VPI24, VPI24_DESC,
			  SIG_DESC_SET(SCU88, 13), COND2);
PIN_DECL_1(Y4, GPIOO5, VPIR3);

#define AA4 118
SIG_EXPR_LIST_DECL_SINGLE(AA4, VPIR4, VPI24, VPI24_DESC,
			  SIG_DESC_SET(SCU88, 14), COND2);
PIN_DECL_1(AA4, GPIOO6, VPIR4);

#define W4 119
SIG_EXPR_LIST_DECL_SINGLE(W4, VPIR5, VPI24, VPI24_DESC,
			  SIG_DESC_SET(SCU88, 15), COND2);
PIN_DECL_1(W4, GPIOO7, VPIR5);

#define V4 120
SIG_EXPR_LIST_DECL_SINGLE(V4, VPIR6, VPI24, VPI24_DESC,
			  SIG_DESC_SET(SCU88, 16), COND2);
PIN_DECL_1(V4, GPIOP0, VPIR6);

#define W5 121
SIG_EXPR_LIST_DECL_SINGLE(W5, VPIR7, VPI24, VPI24_DESC,
			  SIG_DESC_SET(SCU88, 17), COND2);
PIN_DECL_1(W5, GPIOP1, VPIR7);

#define AA5 122
SIG_EXPR_LIST_DECL_SINGLE(AA5, VPIR8, VPI24, VPI24_DESC,
			  SIG_DESC_SET(SCU88, 18), COND2);
PIN_DECL_1(AA5, GPIOP2, VPIR8);

#define AB5 123
SIG_EXPR_LIST_DECL_SINGLE(AB5, VPIR9, VPI24, VPI24_DESC,
			  SIG_DESC_SET(SCU88, 19), COND2);
PIN_DECL_1(AB5, GPIOP3, VPIR9);

FUNC_GROUP_DECL(VPI24, T1, U2, P4, P3, Y1, AB2, AA1, Y2, AA2, P5, R5, T5, V3,
		U3, W3, AA3, Y3, T4, U5, U4, AB3, Y4, AA4, W4, V4, W5, AA5,
		AB5);

#define Y6 124
SIG_EXPR_LIST_DECL_SINGLE(Y6, DASHY6, DASHY6, SIG_DESC_SET(SCU90, 28),
			  SIG_DESC_SET(SCU88, 20));
PIN_DECL_1(Y6, GPIOP4, DASHY6);

#define Y5 125
SIG_EXPR_LIST_DECL_SINGLE(Y5, DASHY5, DASHY5, SIG_DESC_SET(SCU90, 28),
			  SIG_DESC_SET(SCU88, 21));
PIN_DECL_1(Y5, GPIOP5, DASHY5);

#define W6 126
SIG_EXPR_LIST_DECL_SINGLE(W6, DASHW6, DASHW6, SIG_DESC_SET(SCU90, 28),
			  SIG_DESC_SET(SCU88, 22));
PIN_DECL_1(W6, GPIOP6, DASHW6);

#define V6 127
SIG_EXPR_LIST_DECL_SINGLE(V6, DASHV6, DASHV6, SIG_DESC_SET(SCU90, 28),
			  SIG_DESC_SET(SCU88, 23));
PIN_DECL_1(V6, GPIOP7, DASHV6);

#define I2C3_DESC	SIG_DESC_SET(SCU90, 16)

#define A11 128
SIG_EXPR_LIST_DECL_SINGLE(A11, SCL3, I2C3, I2C3_DESC);
PIN_DECL_1(A11, GPIOQ0, SCL3);

#define A10 129
SIG_EXPR_LIST_DECL_SINGLE(A10, SDA3, I2C3, I2C3_DESC);
PIN_DECL_1(A10, GPIOQ1, SDA3);

FUNC_GROUP_DECL(I2C3, A11, A10);

#define I2C4_DESC	SIG_DESC_SET(SCU90, 17)

#define A9 130
SIG_EXPR_LIST_DECL_SINGLE(A9, SCL4, I2C4, I2C4_DESC);
PIN_DECL_1(A9, GPIOQ2, SCL4);

#define B9 131
SIG_EXPR_LIST_DECL_SINGLE(B9, SDA4, I2C4, I2C4_DESC);
PIN_DECL_1(B9, GPIOQ3, SDA4);

FUNC_GROUP_DECL(I2C4, A9, B9);

#define I2C14_DESC	SIG_DESC_SET(SCU90, 27)

#define N21 132
SIG_EXPR_LIST_DECL_SINGLE(N21, SCL14, I2C14, I2C14_DESC);
PIN_DECL_1(N21, GPIOQ4, SCL14);

#define N22 133
SIG_EXPR_LIST_DECL_SINGLE(N22, SDA14, I2C14, I2C14_DESC);
PIN_DECL_1(N22, GPIOQ5, SDA14);

FUNC_GROUP_DECL(I2C14, N21, N22);

#define B10 134
SSSF_PIN_DECL(B10, GPIOQ6, OSCCLK, SIG_DESC_SET(SCU2C, 1));

#define N20 135
SSSF_PIN_DECL(N20, GPIOQ7, PEWAKE, SIG_DESC_SET(SCU2C, 29));

#define AA19 136
SSSF_PIN_DECL(AA19, GPIOR0, FWSPICS1, SIG_DESC_SET(SCU88, 24), COND2);

#define T19 137
SSSF_PIN_DECL(T19, GPIOR1, FWSPICS2, SIG_DESC_SET(SCU88, 25), COND2);

#define T17 138
SSSF_PIN_DECL(T17, GPIOR2, SPI2CS0, SIG_DESC_SET(SCU88, 26), COND2);

#define Y19 139
SSSF_PIN_DECL(Y19, GPIOR3, SPI2CK, SIG_DESC_SET(SCU88, 27), COND2);

#define W19 140
SSSF_PIN_DECL(W19, GPIOR4, SPI2MOSI, SIG_DESC_SET(SCU88, 28), COND2);

#define V19 141
SSSF_PIN_DECL(V19, GPIOR5, SPI2MISO, SIG_DESC_SET(SCU88, 29), COND2);

#define D8 142
SIG_EXPR_LIST_DECL_SINGLE(D8, MDC1, MDIO1, SIG_DESC_SET(SCU88, 30));
PIN_DECL_1(D8, GPIOR6, MDC1);

#define E10 143
SIG_EXPR_LIST_DECL_SINGLE(E10, MDIO1, MDIO1, SIG_DESC_SET(SCU88, 31));
PIN_DECL_1(E10, GPIOR7, MDIO1);

FUNC_GROUP_DECL(MDIO1, D8, E10);

#define VPOOFF0_DESC	{ ASPEED_IP_SCU, SCU94, GENMASK(1, 0), 0, 0 }
#define VPO_DESC	{ ASPEED_IP_SCU, SCU94, GENMASK(1, 0), 1, 0 }
#define VPOOFF1_DESC	{ ASPEED_IP_SCU, SCU94, GENMASK(1, 0), 2, 0 }
#define VPOOFF2_DESC	{ ASPEED_IP_SCU, SCU94, GENMASK(1, 0), 3, 0 }

#define CRT_DVO_EN_DESC	SIG_DESC_IP_SET(ASPEED_IP_GFX, GFX064, 7)

#define V20 144
#define V20_DESC	SIG_DESC_SET(SCU8C, 0)
SIG_EXPR_DECL_SINGLE(VPOB2, VPO, V20_DESC, VPO_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOB2, VPOOFF1, V20_DESC, VPOOFF1_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOB2, VPOOFF2, V20_DESC, VPOOFF2_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_LIST_DECL(VPOB2, VPO,
		SIG_EXPR_PTR(VPOB2, VPO),
		SIG_EXPR_PTR(VPOB2, VPOOFF1),
		SIG_EXPR_PTR(VPOB2, VPOOFF2));
SIG_EXPR_LIST_ALIAS(V20, VPOB2, VPO);
SIG_EXPR_LIST_DECL_SINGLE(V20, SPI2CS1, SPI2CS1, V20_DESC);
PIN_DECL_2(V20, GPIOS0, VPOB2, SPI2CS1);
FUNC_GROUP_DECL(SPI2CS1, V20);

#define U19 145
#define U19_DESC	SIG_DESC_SET(SCU8C, 1)
SIG_EXPR_DECL_SINGLE(VPOB3, VPO, U19_DESC, VPO_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOB3, VPOOFF1, U19_DESC, VPOOFF1_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOB3, VPOOFF2, U19_DESC, VPOOFF2_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_LIST_DECL(VPOB3, VPO,
		SIG_EXPR_PTR(VPOB3, VPO),
		SIG_EXPR_PTR(VPOB3, VPOOFF1),
		SIG_EXPR_PTR(VPOB3, VPOOFF2));
SIG_EXPR_LIST_ALIAS(U19, VPOB3, VPO);
SIG_EXPR_LIST_DECL_SINGLE(U19, BMCINT, BMCINT, U19_DESC);
PIN_DECL_2(U19, GPIOS1, VPOB3, BMCINT);
FUNC_GROUP_DECL(BMCINT, U19);

#define R18 146
#define R18_DESC	SIG_DESC_SET(SCU8C, 2)
SIG_EXPR_DECL_SINGLE(VPOB4, VPO, R18_DESC, VPO_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOB4, VPOOFF1, R18_DESC, VPOOFF1_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOB4, VPOOFF2, R18_DESC, VPOOFF2_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_LIST_DECL(VPOB4, VPO,
		SIG_EXPR_PTR(VPOB4, VPO),
		SIG_EXPR_PTR(VPOB4, VPOOFF1),
		SIG_EXPR_PTR(VPOB4, VPOOFF2));
SIG_EXPR_LIST_ALIAS(R18, VPOB4, VPO);
SIG_EXPR_LIST_DECL_SINGLE(R18, SALT5, SALT5, R18_DESC);
PIN_DECL_2(R18, GPIOS2, VPOB4, SALT5);
FUNC_GROUP_DECL(SALT5, R18);

#define P18 147
#define P18_DESC	SIG_DESC_SET(SCU8C, 3)
SIG_EXPR_DECL_SINGLE(VPOB5, VPO, P18_DESC, VPO_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOB5, VPOOFF1, P18_DESC, VPOOFF1_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOB5, VPOOFF2, P18_DESC, VPOOFF2_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_LIST_DECL(VPOB5, VPO,
		SIG_EXPR_PTR(VPOB5, VPO),
		SIG_EXPR_PTR(VPOB5, VPOOFF1),
		SIG_EXPR_PTR(VPOB5, VPOOFF2));
SIG_EXPR_LIST_ALIAS(P18, VPOB5, VPO);
SIG_EXPR_LIST_DECL_SINGLE(P18, SALT6, SALT6, P18_DESC);
PIN_DECL_2(P18, GPIOS3, VPOB5, SALT6);
FUNC_GROUP_DECL(SALT6, P18);

#define R19 148
#define R19_DESC	SIG_DESC_SET(SCU8C, 4)
SIG_EXPR_DECL_SINGLE(VPOB6, VPO, R19_DESC, VPO_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOB6, VPOOFF1, R19_DESC, VPOOFF1_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOB6, VPOOFF2, R19_DESC, VPOOFF2_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_LIST_DECL(VPOB6, VPO,
		SIG_EXPR_PTR(VPOB6, VPO),
		SIG_EXPR_PTR(VPOB6, VPOOFF1),
		SIG_EXPR_PTR(VPOB6, VPOOFF2));
SIG_EXPR_LIST_ALIAS(R19, VPOB6, VPO);
PIN_DECL_1(R19, GPIOS4, VPOB6);

#define W20 149
#define W20_DESC	SIG_DESC_SET(SCU8C, 5)
SIG_EXPR_DECL_SINGLE(VPOB7, VPO, W20_DESC, VPO_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOB7, VPOOFF1, W20_DESC, VPOOFF1_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOB7, VPOOFF2, W20_DESC, VPOOFF2_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_LIST_DECL(VPOB7, VPO,
		SIG_EXPR_PTR(VPOB7, VPO),
		SIG_EXPR_PTR(VPOB7, VPOOFF1),
		SIG_EXPR_PTR(VPOB7, VPOOFF2));
SIG_EXPR_LIST_ALIAS(W20, VPOB7, VPO);
PIN_DECL_1(W20, GPIOS5, VPOB7);

#define U20 150
#define U20_DESC	SIG_DESC_SET(SCU8C, 6)
SIG_EXPR_DECL_SINGLE(VPOB8, VPO, U20_DESC, VPO_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOB8, VPOOFF1, U20_DESC, VPOOFF1_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOB8, VPOOFF2, U20_DESC, VPOOFF2_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_LIST_DECL(VPOB8, VPO,
		SIG_EXPR_PTR(VPOB8, VPO),
		SIG_EXPR_PTR(VPOB8, VPOOFF1),
		SIG_EXPR_PTR(VPOB8, VPOOFF2));
SIG_EXPR_LIST_ALIAS(U20, VPOB8, VPO);
PIN_DECL_1(U20, GPIOS6, VPOB8);

#define AA20 151
#define AA20_DESC	SIG_DESC_SET(SCU8C, 7)
SIG_EXPR_DECL_SINGLE(VPOB9, VPO, AA20_DESC, VPO_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOB9, VPOOFF1, AA20_DESC, VPOOFF1_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOB9, VPOOFF2, AA20_DESC, VPOOFF2_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_LIST_DECL(VPOB9, VPO,
		SIG_EXPR_PTR(VPOB9, VPO),
		SIG_EXPR_PTR(VPOB9, VPOOFF1),
		SIG_EXPR_PTR(VPOB9, VPOOFF2));
SIG_EXPR_LIST_ALIAS(AA20, VPOB9, VPO);
PIN_DECL_1(AA20, GPIOS7, VPOB9);

/* RGMII1/RMII1 */

#define RMII1_DESC      SIG_DESC_BIT(HW_STRAP1, 6, 0)
#define RMII2_DESC      SIG_DESC_BIT(HW_STRAP1, 7, 0)

#define B5 152
SIG_EXPR_LIST_DECL_SINGLE(B5, GPIOT0, GPIOT0, SIG_DESC_SET(SCUA0, 0));
SIG_EXPR_LIST_DECL_SINGLE(B5, RMII1RCLKO, RMII1, RMII1_DESC,
		SIG_DESC_SET(SCU48, 29));
SIG_EXPR_LIST_DECL_SINGLE(B5, RGMII1TXCK, RGMII1);
PIN_DECL_(B5, SIG_EXPR_LIST_PTR(B5, GPIOT0), SIG_EXPR_LIST_PTR(B5, RMII1RCLKO),
		SIG_EXPR_LIST_PTR(B5, RGMII1TXCK));

#define E9 153
SIG_EXPR_LIST_DECL_SINGLE(E9, GPIOT1, GPIOT1, SIG_DESC_SET(SCUA0, 1));
SIG_EXPR_LIST_DECL_SINGLE(E9, RMII1TXEN, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(E9, RGMII1TXCTL, RGMII1);
PIN_DECL_(E9, SIG_EXPR_LIST_PTR(E9, GPIOT1), SIG_EXPR_LIST_PTR(E9, RMII1TXEN),
		SIG_EXPR_LIST_PTR(E9, RGMII1TXCTL));

#define F9 154
SIG_EXPR_LIST_DECL_SINGLE(F9, GPIOT2, GPIOT2, SIG_DESC_SET(SCUA0, 2));
SIG_EXPR_LIST_DECL_SINGLE(F9, RMII1TXD0, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(F9, RGMII1TXD0, RGMII1);
PIN_DECL_(F9, SIG_EXPR_LIST_PTR(F9, GPIOT2), SIG_EXPR_LIST_PTR(F9, RMII1TXD0),
		SIG_EXPR_LIST_PTR(F9, RGMII1TXD0));

#define A5 155
SIG_EXPR_LIST_DECL_SINGLE(A5, GPIOT3, GPIOT3, SIG_DESC_SET(SCUA0, 3));
SIG_EXPR_LIST_DECL_SINGLE(A5, RMII1TXD1, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(A5, RGMII1TXD1, RGMII1);
PIN_DECL_(A5, SIG_EXPR_LIST_PTR(A5, GPIOT3), SIG_EXPR_LIST_PTR(A5, RMII1TXD1),
		SIG_EXPR_LIST_PTR(A5, RGMII1TXD1));

#define E7 156
SIG_EXPR_LIST_DECL_SINGLE(E7, GPIOT4, GPIOT4, SIG_DESC_SET(SCUA0, 4));
SIG_EXPR_LIST_DECL_SINGLE(E7, RMII1DASH0, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(E7, RGMII1TXD2, RGMII1);
PIN_DECL_(E7, SIG_EXPR_LIST_PTR(E7, GPIOT4), SIG_EXPR_LIST_PTR(E7, RMII1DASH0),
		SIG_EXPR_LIST_PTR(E7, RGMII1TXD2));

#define D7 157
SIG_EXPR_LIST_DECL_SINGLE(D7, GPIOT5, GPIOT5, SIG_DESC_SET(SCUA0, 5));
SIG_EXPR_LIST_DECL_SINGLE(D7, RMII1DASH1, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(D7, RGMII1TXD3, RGMII1);
PIN_DECL_(D7, SIG_EXPR_LIST_PTR(D7, GPIOT5), SIG_EXPR_LIST_PTR(D7, RMII1DASH1),
		SIG_EXPR_LIST_PTR(D7, RGMII1TXD3));

#define B2 158
SIG_EXPR_LIST_DECL_SINGLE(B2, GPIOT6, GPIOT6, SIG_DESC_SET(SCUA0, 6));
SIG_EXPR_LIST_DECL_SINGLE(B2, RMII2RCLKO, RMII2, RMII2_DESC,
		SIG_DESC_SET(SCU48, 30));
SIG_EXPR_LIST_DECL_SINGLE(B2, RGMII2TXCK, RGMII2);
PIN_DECL_(B2, SIG_EXPR_LIST_PTR(B2, GPIOT6), SIG_EXPR_LIST_PTR(B2, RMII2RCLKO),
		SIG_EXPR_LIST_PTR(B2, RGMII2TXCK));

#define B1 159
SIG_EXPR_LIST_DECL_SINGLE(B1, GPIOT7, GPIOT7, SIG_DESC_SET(SCUA0, 7));
SIG_EXPR_LIST_DECL_SINGLE(B1, RMII2TXEN, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(B1, RGMII2TXCTL, RGMII2);
PIN_DECL_(B1, SIG_EXPR_LIST_PTR(B1, GPIOT7), SIG_EXPR_LIST_PTR(B1, RMII2TXEN),
		SIG_EXPR_LIST_PTR(B1, RGMII2TXCTL));

#define A2 160
SIG_EXPR_LIST_DECL_SINGLE(A2, GPIOU0, GPIOU0, SIG_DESC_SET(SCUA0, 8));
SIG_EXPR_LIST_DECL_SINGLE(A2, RMII2TXD0, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(A2, RGMII2TXD0, RGMII2);
PIN_DECL_(A2, SIG_EXPR_LIST_PTR(A2, GPIOU0), SIG_EXPR_LIST_PTR(A2, RMII2TXD0),
		SIG_EXPR_LIST_PTR(A2, RGMII2TXD0));

#define B3 161
SIG_EXPR_LIST_DECL_SINGLE(B3, GPIOU1, GPIOU1, SIG_DESC_SET(SCUA0, 9));
SIG_EXPR_LIST_DECL_SINGLE(B3, RMII2TXD1, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(B3, RGMII2TXD1, RGMII2);
PIN_DECL_(B3, SIG_EXPR_LIST_PTR(B3, GPIOU1), SIG_EXPR_LIST_PTR(B3, RMII2TXD1),
		SIG_EXPR_LIST_PTR(B3, RGMII2TXD1));

#define D5 162
SIG_EXPR_LIST_DECL_SINGLE(D5, GPIOU2, GPIOU2, SIG_DESC_SET(SCUA0, 10));
SIG_EXPR_LIST_DECL_SINGLE(D5, RMII2DASH0, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(D5, RGMII2TXD2, RGMII2);
PIN_DECL_(D5, SIG_EXPR_LIST_PTR(D5, GPIOU2), SIG_EXPR_LIST_PTR(D5, RMII2DASH0),
		SIG_EXPR_LIST_PTR(D5, RGMII2TXD2));

#define D4 163
SIG_EXPR_LIST_DECL_SINGLE(D4, GPIOU3, GPIOU3, SIG_DESC_SET(SCUA0, 11));
SIG_EXPR_LIST_DECL_SINGLE(D4, RMII2DASH1, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(D4, RGMII2TXD3, RGMII2);
PIN_DECL_(D4, SIG_EXPR_LIST_PTR(D4, GPIOU3), SIG_EXPR_LIST_PTR(D4, RMII2DASH1),
		SIG_EXPR_LIST_PTR(D4, RGMII2TXD3));

#define B4 164
SIG_EXPR_LIST_DECL_SINGLE(B4, GPIOU4, GPIOU4, SIG_DESC_SET(SCUA0, 12));
SIG_EXPR_LIST_DECL_SINGLE(B4, RMII1RCLKI, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(B4, RGMII1RXCK, RGMII1);
PIN_DECL_(B4, SIG_EXPR_LIST_PTR(B4, GPIOU4), SIG_EXPR_LIST_PTR(B4, RMII1RCLKI),
		SIG_EXPR_LIST_PTR(B4, RGMII1RXCK));

#define A4 165
SIG_EXPR_LIST_DECL_SINGLE(A4, GPIOU5, GPIOU5, SIG_DESC_SET(SCUA0, 13));
SIG_EXPR_LIST_DECL_SINGLE(A4, RMII1DASH2, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(A4, RGMII1RXCTL, RGMII1);
PIN_DECL_(A4, SIG_EXPR_LIST_PTR(A4, GPIOU5), SIG_EXPR_LIST_PTR(A4, RMII1DASH2),
		SIG_EXPR_LIST_PTR(A4, RGMII1RXCTL));

#define A3 166
SIG_EXPR_LIST_DECL_SINGLE(A3, GPIOU6, GPIOU6, SIG_DESC_SET(SCUA0, 14));
SIG_EXPR_LIST_DECL_SINGLE(A3, RMII1RXD0, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(A3, RGMII1RXD0, RGMII1);
PIN_DECL_(A3, SIG_EXPR_LIST_PTR(A3, GPIOU6), SIG_EXPR_LIST_PTR(A3, RMII1RXD0),
		SIG_EXPR_LIST_PTR(A3, RGMII1RXD0));

#define D6 167
SIG_EXPR_LIST_DECL_SINGLE(D6, GPIOU7, GPIOU7, SIG_DESC_SET(SCUA0, 15));
SIG_EXPR_LIST_DECL_SINGLE(D6, RMII1RXD1, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(D6, RGMII1RXD1, RGMII1);
PIN_DECL_(D6, SIG_EXPR_LIST_PTR(D6, GPIOU7), SIG_EXPR_LIST_PTR(D6, RMII1RXD1),
		SIG_EXPR_LIST_PTR(D6, RGMII1RXD1));

#define C5 168
SIG_EXPR_LIST_DECL_SINGLE(C5, GPIOV0, GPIOV0, SIG_DESC_SET(SCUA0, 16));
SIG_EXPR_LIST_DECL_SINGLE(C5, RMII1CRSDV, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(C5, RGMII1RXD2, RGMII1);
PIN_DECL_(C5, SIG_EXPR_LIST_PTR(C5, GPIOV0), SIG_EXPR_LIST_PTR(C5, RMII1CRSDV),
		SIG_EXPR_LIST_PTR(C5, RGMII1RXD2));

#define C4 169
SIG_EXPR_LIST_DECL_SINGLE(C4, GPIOV1, GPIOV1, SIG_DESC_SET(SCUA0, 17));
SIG_EXPR_LIST_DECL_SINGLE(C4, RMII1RXER, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(C4, RGMII1RXD3, RGMII1);
PIN_DECL_(C4, SIG_EXPR_LIST_PTR(C4, GPIOV1), SIG_EXPR_LIST_PTR(C4, RMII1RXER),
		SIG_EXPR_LIST_PTR(C4, RGMII1RXD3));

FUNC_GROUP_DECL(RGMII1, B4, A4, A3, D6, C5, C4, B5, E9, F9, A5, E7, D7);
FUNC_GROUP_DECL(RMII1, B4, A3, D6, C5, C4, B5, E9, F9, A5);

#define C2 170
SIG_EXPR_LIST_DECL_SINGLE(C2, GPIOV2, GPIOV2, SIG_DESC_SET(SCUA0, 18));
SIG_EXPR_LIST_DECL_SINGLE(C2, RMII2RCLKI, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(C2, RGMII2RXCK, RGMII2);
PIN_DECL_(C2, SIG_EXPR_LIST_PTR(C2, GPIOV2), SIG_EXPR_LIST_PTR(C2, RMII2RCLKI),
		SIG_EXPR_LIST_PTR(C2, RGMII2RXCK));

#define C1 171
SIG_EXPR_LIST_DECL_SINGLE(C1, GPIOV3, GPIOV3, SIG_DESC_SET(SCUA0, 19));
SIG_EXPR_LIST_DECL_SINGLE(C1, RMII2DASH2, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(C1, RGMII2RXCTL, RGMII2);
PIN_DECL_(C1, SIG_EXPR_LIST_PTR(C1, GPIOV3), SIG_EXPR_LIST_PTR(C1, RMII2DASH2),
		SIG_EXPR_LIST_PTR(C1, RGMII2RXCTL));

#define C3 172
SIG_EXPR_LIST_DECL_SINGLE(C3, GPIOV4, GPIOV4, SIG_DESC_SET(SCUA0, 20));
SIG_EXPR_LIST_DECL_SINGLE(C3, RMII2RXD0, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(C3, RGMII2RXD0, RGMII2);
PIN_DECL_(C3, SIG_EXPR_LIST_PTR(C3, GPIOV4), SIG_EXPR_LIST_PTR(C3, RMII2RXD0),
		SIG_EXPR_LIST_PTR(C3, RGMII2RXD0));

#define D1 173
SIG_EXPR_LIST_DECL_SINGLE(D1, GPIOV5, GPIOV5, SIG_DESC_SET(SCUA0, 21));
SIG_EXPR_LIST_DECL_SINGLE(D1, RMII2RXD1, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(D1, RGMII2RXD1, RGMII2);
PIN_DECL_(D1, SIG_EXPR_LIST_PTR(D1, GPIOV5), SIG_EXPR_LIST_PTR(D1, RMII2RXD1),
		SIG_EXPR_LIST_PTR(D1, RGMII2RXD1));

#define D2 174
SIG_EXPR_LIST_DECL_SINGLE(D2, GPIOV6, GPIOV6, SIG_DESC_SET(SCUA0, 22));
SIG_EXPR_LIST_DECL_SINGLE(D2, RMII2CRSDV, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(D2, RGMII2RXD2, RGMII2);
PIN_DECL_(D2, SIG_EXPR_LIST_PTR(D2, GPIOV6), SIG_EXPR_LIST_PTR(D2, RMII2CRSDV),
		SIG_EXPR_LIST_PTR(D2, RGMII2RXD2));

#define E6 175
SIG_EXPR_LIST_DECL_SINGLE(E6, GPIOV7, GPIOV7, SIG_DESC_SET(SCUA0, 23));
SIG_EXPR_LIST_DECL_SINGLE(E6, RMII2RXER, RMII2, RMII2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(E6, RGMII2RXD3, RGMII2);
PIN_DECL_(E6, SIG_EXPR_LIST_PTR(E6, GPIOV7), SIG_EXPR_LIST_PTR(E6, RMII2RXER),
		SIG_EXPR_LIST_PTR(E6, RGMII2RXD3));

FUNC_GROUP_DECL(RGMII2, B2, B1, A2, B3, D5, D4, C2, C1, C3, D1, D2, E6);
FUNC_GROUP_DECL(RMII2, B2, B1, A2, B3, C2, C3, D1, D2, E6);

#define F4 176
SIG_EXPR_LIST_DECL_SINGLE(F4, GPIOW0, GPIOW0, SIG_DESC_SET(SCUA0, 24));
SIG_EXPR_LIST_DECL_SINGLE(F4, ADC0, ADC0);
PIN_DECL_(F4, SIG_EXPR_LIST_PTR(F4, GPIOW0), SIG_EXPR_LIST_PTR(F4, ADC0));
FUNC_GROUP_DECL(ADC0, F4);

#define F5 177
SIG_EXPR_LIST_DECL_SINGLE(F5, GPIOW1, GPIOW1, SIG_DESC_SET(SCUA0, 25));
SIG_EXPR_LIST_DECL_SINGLE(F5, ADC1, ADC1);
PIN_DECL_(F5, SIG_EXPR_LIST_PTR(F5, GPIOW1), SIG_EXPR_LIST_PTR(F5, ADC1));
FUNC_GROUP_DECL(ADC1, F5);

#define E2 178
SIG_EXPR_LIST_DECL_SINGLE(E2, GPIOW2, GPIOW2, SIG_DESC_SET(SCUA0, 26));
SIG_EXPR_LIST_DECL_SINGLE(E2, ADC2, ADC2);
PIN_DECL_(E2, SIG_EXPR_LIST_PTR(E2, GPIOW2), SIG_EXPR_LIST_PTR(E2, ADC2));
FUNC_GROUP_DECL(ADC2, E2);

#define E1 179
SIG_EXPR_LIST_DECL_SINGLE(E1, GPIOW3, GPIOW3, SIG_DESC_SET(SCUA0, 27));
SIG_EXPR_LIST_DECL_SINGLE(E1, ADC3, ADC3);
PIN_DECL_(E1, SIG_EXPR_LIST_PTR(E1, GPIOW3), SIG_EXPR_LIST_PTR(E1, ADC3));
FUNC_GROUP_DECL(ADC3, E1);

#define F3 180
SIG_EXPR_LIST_DECL_SINGLE(F3, GPIOW4, GPIOW4, SIG_DESC_SET(SCUA0, 28));
SIG_EXPR_LIST_DECL_SINGLE(F3, ADC4, ADC4);
PIN_DECL_(F3, SIG_EXPR_LIST_PTR(F3, GPIOW4), SIG_EXPR_LIST_PTR(F3, ADC4));
FUNC_GROUP_DECL(ADC4, F3);

#define E3 181
SIG_EXPR_LIST_DECL_SINGLE(E3, GPIOW5, GPIOW5, SIG_DESC_SET(SCUA0, 29));
SIG_EXPR_LIST_DECL_SINGLE(E3, ADC5, ADC5);
PIN_DECL_(E3, SIG_EXPR_LIST_PTR(E3, GPIOW5), SIG_EXPR_LIST_PTR(E3, ADC5));
FUNC_GROUP_DECL(ADC5, E3);

#define G5 182
SIG_EXPR_LIST_DECL_SINGLE(G5, GPIOW6, GPIOW6, SIG_DESC_SET(SCUA0, 30));
SIG_EXPR_LIST_DECL_SINGLE(G5, ADC6, ADC6);
PIN_DECL_(G5, SIG_EXPR_LIST_PTR(G5, GPIOW6), SIG_EXPR_LIST_PTR(G5, ADC6));
FUNC_GROUP_DECL(ADC6, G5);

#define G4 183
SIG_EXPR_LIST_DECL_SINGLE(G4, GPIOW7, GPIOW7, SIG_DESC_SET(SCUA0, 31));
SIG_EXPR_LIST_DECL_SINGLE(G4, ADC7, ADC7);
PIN_DECL_(G4, SIG_EXPR_LIST_PTR(G4, GPIOW7), SIG_EXPR_LIST_PTR(G4, ADC7));
FUNC_GROUP_DECL(ADC7, G4);

#define F2 184
SIG_EXPR_LIST_DECL_SINGLE(F2, GPIOX0, GPIOX0, SIG_DESC_SET(SCUA4, 0));
SIG_EXPR_LIST_DECL_SINGLE(F2, ADC8, ADC8);
PIN_DECL_(F2, SIG_EXPR_LIST_PTR(F2, GPIOX0), SIG_EXPR_LIST_PTR(F2, ADC8));
FUNC_GROUP_DECL(ADC8, F2);

#define G3 185
SIG_EXPR_LIST_DECL_SINGLE(G3, GPIOX1, GPIOX1, SIG_DESC_SET(SCUA4, 1));
SIG_EXPR_LIST_DECL_SINGLE(G3, ADC9, ADC9);
PIN_DECL_(G3, SIG_EXPR_LIST_PTR(G3, GPIOX1), SIG_EXPR_LIST_PTR(G3, ADC9));
FUNC_GROUP_DECL(ADC9, G3);

#define G2 186
SIG_EXPR_LIST_DECL_SINGLE(G2, GPIOX2, GPIOX2, SIG_DESC_SET(SCUA4, 2));
SIG_EXPR_LIST_DECL_SINGLE(G2, ADC10, ADC10);
PIN_DECL_(G2, SIG_EXPR_LIST_PTR(G2, GPIOX2), SIG_EXPR_LIST_PTR(G2, ADC10));
FUNC_GROUP_DECL(ADC10, G2);

#define F1 187
SIG_EXPR_LIST_DECL_SINGLE(F1, GPIOX3, GPIOX3, SIG_DESC_SET(SCUA4, 3));
SIG_EXPR_LIST_DECL_SINGLE(F1, ADC11, ADC11);
PIN_DECL_(F1, SIG_EXPR_LIST_PTR(F1, GPIOX3), SIG_EXPR_LIST_PTR(F1, ADC11));
FUNC_GROUP_DECL(ADC11, F1);

#define H5 188
SIG_EXPR_LIST_DECL_SINGLE(H5, GPIOX4, GPIOX4, SIG_DESC_SET(SCUA4, 4));
SIG_EXPR_LIST_DECL_SINGLE(H5, ADC12, ADC12);
PIN_DECL_(H5, SIG_EXPR_LIST_PTR(H5, GPIOX4), SIG_EXPR_LIST_PTR(H5, ADC12));
FUNC_GROUP_DECL(ADC12, H5);

#define G1 189
SIG_EXPR_LIST_DECL_SINGLE(G1, GPIOX5, GPIOX5, SIG_DESC_SET(SCUA4, 5));
SIG_EXPR_LIST_DECL_SINGLE(G1, ADC13, ADC13);
PIN_DECL_(G1, SIG_EXPR_LIST_PTR(G1, GPIOX5), SIG_EXPR_LIST_PTR(G1, ADC13));
FUNC_GROUP_DECL(ADC13, G1);

#define H3 190
SIG_EXPR_LIST_DECL_SINGLE(H3, GPIOX6, GPIOX6, SIG_DESC_SET(SCUA4, 6));
SIG_EXPR_LIST_DECL_SINGLE(H3, ADC14, ADC14);
PIN_DECL_(H3, SIG_EXPR_LIST_PTR(H3, GPIOX6), SIG_EXPR_LIST_PTR(H3, ADC14));
FUNC_GROUP_DECL(ADC14, H3);

#define H4 191
SIG_EXPR_LIST_DECL_SINGLE(H4, GPIOX7, GPIOX7, SIG_DESC_SET(SCUA4, 7));
SIG_EXPR_LIST_DECL_SINGLE(H4, ADC15, ADC15);
PIN_DECL_(H4, SIG_EXPR_LIST_PTR(H4, GPIOX7), SIG_EXPR_LIST_PTR(H4, ADC15));
FUNC_GROUP_DECL(ADC15, H4);

#define ACPI_DESC	SIG_DESC_SET(HW_STRAP1, 19)

#define R22 192
SIG_EXPR_DECL_SINGLE(SIOS3, SIOS3, SIG_DESC_SET(SCUA4, 8));
SIG_EXPR_DECL_SINGLE(SIOS3, ACPI, ACPI_DESC);
SIG_EXPR_LIST_DECL_DUAL(R22, SIOS3, SIOS3, ACPI);
SIG_EXPR_LIST_DECL_SINGLE(R22, DASHR22, DASHR22, SIG_DESC_SET(SCU94, 10));
PIN_DECL_2(R22, GPIOY0, SIOS3, DASHR22);
FUNC_GROUP_DECL(SIOS3, R22);

#define R21 193
SIG_EXPR_DECL_SINGLE(SIOS5, SIOS5, SIG_DESC_SET(SCUA4, 9));
SIG_EXPR_DECL_SINGLE(SIOS5, ACPI, ACPI_DESC);
SIG_EXPR_LIST_DECL_DUAL(R21, SIOS5, SIOS5, ACPI);
SIG_EXPR_LIST_DECL_SINGLE(R21, DASHR21, DASHR21, SIG_DESC_SET(SCU94, 10));
PIN_DECL_2(R21, GPIOY1, SIOS5, DASHR21);
FUNC_GROUP_DECL(SIOS5, R21);

#define P22 194
SIG_EXPR_DECL_SINGLE(SIOPWREQ, SIOPWREQ, SIG_DESC_SET(SCUA4, 10));
SIG_EXPR_DECL_SINGLE(SIOPWREQ, ACPI, ACPI_DESC);
SIG_EXPR_LIST_DECL_DUAL(P22, SIOPWREQ, SIOPWREQ, ACPI);
SIG_EXPR_LIST_DECL_SINGLE(P22, DASHP22, DASHP22, SIG_DESC_SET(SCU94, 11));
PIN_DECL_2(P22, GPIOY2, SIOPWREQ, DASHP22);
FUNC_GROUP_DECL(SIOPWREQ, P22);

#define P21 195
SIG_EXPR_DECL_SINGLE(SIOONCTRL, SIOONCTRL, SIG_DESC_SET(SCUA4, 11));
SIG_EXPR_DECL_SINGLE(SIOONCTRL, ACPI, ACPI_DESC);
SIG_EXPR_LIST_DECL_DUAL(P21, SIOONCTRL, SIOONCTRL, ACPI);
SIG_EXPR_LIST_DECL_SINGLE(P21, DASHP21, DASHP21, SIG_DESC_SET(SCU94, 11));
PIN_DECL_2(P21, GPIOY3, SIOONCTRL, DASHP21);
FUNC_GROUP_DECL(SIOONCTRL, P21);

#define M18 196
SSSF_PIN_DECL(M18, GPIOY4, SCL1, SIG_DESC_SET(SCUA4, 12));

#define M19 197
SSSF_PIN_DECL(M19, GPIOY5, SDA1, SIG_DESC_SET(SCUA4, 13));

#define M20 198
SSSF_PIN_DECL(M20, GPIOY6, SCL2, SIG_DESC_SET(SCUA4, 14));

#define P20 199
SSSF_PIN_DECL(P20, GPIOY7, SDA2, SIG_DESC_SET(SCUA4, 15));

#define PNOR_DESC	SIG_DESC_SET(SCU90, 31)

#define Y20 200
#define Y20_DESC	SIG_DESC_SET(SCUA4, 16)
SIG_EXPR_DECL_SINGLE(VPOG2, VPO, Y20_DESC, VPO_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOG2, VPOOFF1, Y20_DESC, VPOOFF1_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOG2, VPOOFF2, Y20_DESC, VPOOFF2_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_LIST_DECL(VPOG2, VPO,
		SIG_EXPR_PTR(VPOG2, VPO),
		SIG_EXPR_PTR(VPOG2, VPOOFF1),
		SIG_EXPR_PTR(VPOG2, VPOOFF2));
SIG_EXPR_LIST_ALIAS(Y20, VPOG2, VPO);
SIG_EXPR_DECL_SINGLE(SIOPBI, SIOPBI, Y20_DESC);
SIG_EXPR_DECL_SINGLE(SIOPBI, ACPI, Y20_DESC);
SIG_EXPR_LIST_DECL_DUAL(Y20, SIOPBI, SIOPBI, ACPI);
SIG_EXPR_LIST_DECL_SINGLE(Y20, NORA0, PNOR, PNOR_DESC);
SIG_EXPR_LIST_DECL_SINGLE(Y20, GPIOZ0, GPIOZ0);
PIN_DECL_(Y20, SIG_EXPR_LIST_PTR(Y20, VPOG2), SIG_EXPR_LIST_PTR(Y20, SIOPBI),
		SIG_EXPR_LIST_PTR(Y20, NORA0), SIG_EXPR_LIST_PTR(Y20, GPIOZ0));
FUNC_GROUP_DECL(SIOPBI, Y20);

#define AB20 201
#define AB20_DESC	SIG_DESC_SET(SCUA4, 17)
SIG_EXPR_DECL_SINGLE(VPOG3, VPO, AB20_DESC, VPO_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOG3, VPOOFF1, AB20_DESC, VPOOFF1_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOG3, VPOOFF2, AB20_DESC, VPOOFF2_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_LIST_DECL(VPOG3, VPO,
		SIG_EXPR_PTR(VPOG3, VPO),
		SIG_EXPR_PTR(VPOG3, VPOOFF1),
		SIG_EXPR_PTR(VPOG3, VPOOFF2));
SIG_EXPR_LIST_ALIAS(AB20, VPOG3, VPO);
SIG_EXPR_DECL_SINGLE(SIOPWRGD, SIOPWRGD, AB20_DESC);
SIG_EXPR_DECL_SINGLE(SIOPWRGD, ACPI, AB20_DESC);
SIG_EXPR_LIST_DECL_DUAL(AB20, SIOPWRGD, SIOPWRGD, ACPI);
SIG_EXPR_LIST_DECL_SINGLE(AB20, NORA1, PNOR, PNOR_DESC);
SIG_EXPR_LIST_DECL_SINGLE(AB20, GPIOZ1, GPIOZ1);
PIN_DECL_(AB20, SIG_EXPR_LIST_PTR(AB20, VPOG3),
	  SIG_EXPR_LIST_PTR(AB20, SIOPWRGD), SIG_EXPR_LIST_PTR(AB20, NORA1),
	  SIG_EXPR_LIST_PTR(AB20, GPIOZ1));
FUNC_GROUP_DECL(SIOPWRGD, AB20);

#define AB21 202
#define AB21_DESC	SIG_DESC_SET(SCUA4, 18)
SIG_EXPR_DECL_SINGLE(VPOG4, VPO, AB21_DESC, VPO_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOG4, VPOOFF1, AB21_DESC, VPOOFF1_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOG4, VPOOFF2, AB21_DESC, VPOOFF2_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_LIST_DECL(VPOG4, VPO,
		SIG_EXPR_PTR(VPOG4, VPO),
		SIG_EXPR_PTR(VPOG4, VPOOFF1),
		SIG_EXPR_PTR(VPOG4, VPOOFF2));
SIG_EXPR_LIST_ALIAS(AB21, VPOG4, VPO);
SIG_EXPR_DECL_SINGLE(SIOPBO, SIOPBO, AB21_DESC);
SIG_EXPR_DECL_SINGLE(SIOPBO, ACPI, AB21_DESC);
SIG_EXPR_LIST_DECL_DUAL(AB21, SIOPBO, SIOPBO, ACPI);
SIG_EXPR_LIST_DECL_SINGLE(AB21, NORA2, PNOR, PNOR_DESC);
SIG_EXPR_LIST_DECL_SINGLE(AB21, GPIOZ2, GPIOZ2);
PIN_DECL_(AB21, SIG_EXPR_LIST_PTR(AB21, VPOG4),
	  SIG_EXPR_LIST_PTR(AB21, SIOPBO), SIG_EXPR_LIST_PTR(AB21, NORA2),
	  SIG_EXPR_LIST_PTR(AB21, GPIOZ2));
FUNC_GROUP_DECL(SIOPBO, AB21);

#define AA21 203
#define AA21_DESC	SIG_DESC_SET(SCUA4, 19)
SIG_EXPR_DECL_SINGLE(VPOG5, VPO, AA21_DESC, VPO_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOG5, VPOOFF1, AA21_DESC, VPOOFF1_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOG5, VPOOFF2, AA21_DESC, VPOOFF2_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_LIST_DECL(VPOG5, VPO,
		SIG_EXPR_PTR(VPOG5, VPO),
		SIG_EXPR_PTR(VPOG5, VPOOFF1),
		SIG_EXPR_PTR(VPOG5, VPOOFF2));
SIG_EXPR_LIST_ALIAS(AA21, VPOG5, VPO);
SIG_EXPR_DECL_SINGLE(SIOSCI, SIOSCI, AA21_DESC);
SIG_EXPR_DECL_SINGLE(SIOSCI, ACPI, AA21_DESC);
SIG_EXPR_LIST_DECL_DUAL(AA21, SIOSCI, SIOSCI, ACPI);
SIG_EXPR_LIST_DECL_SINGLE(AA21, NORA3, PNOR, PNOR_DESC);
SIG_EXPR_LIST_DECL_SINGLE(AA21, GPIOZ3, GPIOZ3);
PIN_DECL_(AA21, SIG_EXPR_LIST_PTR(AA21, VPOG5),
	  SIG_EXPR_LIST_PTR(AA21, SIOSCI), SIG_EXPR_LIST_PTR(AA21, NORA3),
	  SIG_EXPR_LIST_PTR(AA21, GPIOZ3));
FUNC_GROUP_DECL(SIOSCI, AA21);

FUNC_GROUP_DECL(ACPI, R22, R21, P22, P21, Y20, AB20, AB21, AA21);

/* CRT DVO disabled, configured for single-edge mode */
#define CRT_DVO_DS_DESC { ASPEED_IP_GFX, GFX064, GENMASK(7, 6), 0, 0 }

/* CRT DVO disabled, configured for dual-edge mode */
#define CRT_DVO_DD_DESC { ASPEED_IP_GFX, GFX064, GENMASK(7, 6), 1, 1 }

/* CRT DVO enabled, configured for single-edge mode */
#define CRT_DVO_ES_DESC { ASPEED_IP_GFX, GFX064, GENMASK(7, 6), 2, 2 }

/* CRT DVO enabled, configured for dual-edge mode */
#define CRT_DVO_ED_DESC { ASPEED_IP_GFX, GFX064, GENMASK(7, 6), 3, 3 }

#define U21 204
#define U21_DESC	SIG_DESC_SET(SCUA4, 20)
SIG_EXPR_DECL_SINGLE(VPOG6, VPO, U21_DESC, VPO_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOG6, VPOOFF1, U21_DESC, VPOOFF1_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOG6, VPOOFF2, U21_DESC, VPOOFF2_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_LIST_DECL(VPOG6, VPO,
		SIG_EXPR_PTR(VPOG6, VPO),
		SIG_EXPR_PTR(VPOG6, VPOOFF1),
		SIG_EXPR_PTR(VPOG6, VPOOFF2));
SIG_EXPR_LIST_ALIAS(U21, VPOG6, VPO);
SIG_EXPR_LIST_DECL_SINGLE(U21, NORA4, PNOR, PNOR_DESC);
PIN_DECL_2(U21, GPIOZ4, VPOG6, NORA4);

#define W22 205
#define W22_DESC	SIG_DESC_SET(SCUA4, 21)
SIG_EXPR_DECL_SINGLE(VPOG7, VPO, W22_DESC, VPO_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOG7, VPOOFF1, W22_DESC, VPOOFF1_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOG7, VPOOFF2, W22_DESC, VPOOFF2_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_LIST_DECL(VPOG7, VPO,
		SIG_EXPR_PTR(VPOG7, VPO),
		SIG_EXPR_PTR(VPOG7, VPOOFF1),
		SIG_EXPR_PTR(VPOG7, VPOOFF2));
SIG_EXPR_LIST_ALIAS(W22, VPOG7, VPO);
SIG_EXPR_LIST_DECL_SINGLE(W22, NORA5, PNOR, PNOR_DESC);
PIN_DECL_2(W22, GPIOZ5, VPOG7, NORA5);

#define V22 206
#define V22_DESC	SIG_DESC_SET(SCUA4, 22)
SIG_EXPR_DECL_SINGLE(VPOG8, VPO, V22_DESC, VPO_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOG8, VPOOFF1, V22_DESC, VPOOFF1_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOG8, VPOOFF2, V22_DESC, VPOOFF2_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_LIST_DECL(VPOG8, VPO,
		SIG_EXPR_PTR(VPOG8, VPO),
		SIG_EXPR_PTR(VPOG8, VPOOFF1),
		SIG_EXPR_PTR(VPOG8, VPOOFF2));
SIG_EXPR_LIST_ALIAS(V22, VPOG8, VPO);
SIG_EXPR_LIST_DECL_SINGLE(V22, NORA6, PNOR, PNOR_DESC);
PIN_DECL_2(V22, GPIOZ6, VPOG8, NORA6);

#define W21 207
#define W21_DESC	SIG_DESC_SET(SCUA4, 23)
SIG_EXPR_DECL_SINGLE(VPOG9, VPO, W21_DESC, VPO_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOG9, VPOOFF1, W21_DESC, VPOOFF1_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOG9, VPOOFF2, W21_DESC, VPOOFF2_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_LIST_DECL(VPOG9, VPO,
		SIG_EXPR_PTR(VPOG9, VPO),
		SIG_EXPR_PTR(VPOG9, VPOOFF1),
		SIG_EXPR_PTR(VPOG9, VPOOFF2));
SIG_EXPR_LIST_ALIAS(W21, VPOG9, VPO);
SIG_EXPR_LIST_DECL_SINGLE(W21, NORA7, PNOR, PNOR_DESC);
PIN_DECL_2(W21, GPIOZ7, VPOG9, NORA7);

#define Y21 208
#define Y21_DESC	SIG_DESC_SET(SCUA4, 24)
SIG_EXPR_DECL_SINGLE(VPOR2, VPO, Y21_DESC, VPO_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOR2, VPOOFF1, Y21_DESC, VPOOFF1_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOR2, VPOOFF2, Y21_DESC, VPOOFF2_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_LIST_DECL(VPOR2, VPO,
		SIG_EXPR_PTR(VPOR2, VPO),
		SIG_EXPR_PTR(VPOR2, VPOOFF1),
		SIG_EXPR_PTR(VPOR2, VPOOFF2));
SIG_EXPR_LIST_ALIAS(Y21, VPOR2, VPO);
SIG_EXPR_LIST_DECL_SINGLE(Y21, SALT7, SALT7, Y21_DESC);
SIG_EXPR_LIST_DECL_SINGLE(Y21, NORD0, PNOR, PNOR_DESC);
SIG_EXPR_LIST_DECL_SINGLE(Y21, GPIOAA0, GPIOAA0);
PIN_DECL_(Y21, SIG_EXPR_LIST_PTR(Y21, VPOR2), SIG_EXPR_LIST_PTR(Y21, SALT7),
		SIG_EXPR_LIST_PTR(Y21, NORD0), SIG_EXPR_LIST_PTR(Y21, GPIOAA0));
FUNC_GROUP_DECL(SALT7, Y21);

#define V21 209
#define V21_DESC	SIG_DESC_SET(SCUA4, 25)
SIG_EXPR_DECL_SINGLE(VPOR3, VPO, V21_DESC, VPO_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOR3, VPOOFF1, V21_DESC, VPOOFF1_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOR3, VPOOFF2, V21_DESC, VPOOFF2_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_LIST_DECL(VPOR3, VPO,
		SIG_EXPR_PTR(VPOR3, VPO),
		SIG_EXPR_PTR(VPOR3, VPOOFF1),
		SIG_EXPR_PTR(VPOR3, VPOOFF2));
SIG_EXPR_LIST_ALIAS(V21, VPOR3, VPO);
SIG_EXPR_LIST_DECL_SINGLE(V21, SALT8, SALT8, V21_DESC);
SIG_EXPR_LIST_DECL_SINGLE(V21, NORD1, PNOR, PNOR_DESC);
SIG_EXPR_LIST_DECL_SINGLE(V21, GPIOAA1, GPIOAA1);
PIN_DECL_(V21, SIG_EXPR_LIST_PTR(V21, VPOR3), SIG_EXPR_LIST_PTR(V21, SALT8),
		SIG_EXPR_LIST_PTR(V21, NORD1), SIG_EXPR_LIST_PTR(V21, GPIOAA1));
FUNC_GROUP_DECL(SALT8, V21);

#define Y22 210
#define Y22_DESC	SIG_DESC_SET(SCUA4, 26)
SIG_EXPR_DECL_SINGLE(VPOR4, VPO, Y22_DESC, VPO_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOR4, VPOOFF1, Y22_DESC, VPOOFF1_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOR4, VPOOFF2, Y22_DESC, VPOOFF2_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_LIST_DECL(VPOR4, VPO,
		SIG_EXPR_PTR(VPOR4, VPO),
		SIG_EXPR_PTR(VPOR4, VPOOFF1),
		SIG_EXPR_PTR(VPOR4, VPOOFF2));
SIG_EXPR_LIST_ALIAS(Y22, VPOR4, VPO);
SIG_EXPR_LIST_DECL_SINGLE(Y22, SALT9, SALT9, Y22_DESC);
SIG_EXPR_LIST_DECL_SINGLE(Y22, NORD2, PNOR, PNOR_DESC);
SIG_EXPR_LIST_DECL_SINGLE(Y22, GPIOAA2, GPIOAA2);
PIN_DECL_(Y22, SIG_EXPR_LIST_PTR(Y22, VPOR4), SIG_EXPR_LIST_PTR(Y22, SALT9),
		SIG_EXPR_LIST_PTR(Y22, NORD2), SIG_EXPR_LIST_PTR(Y22, GPIOAA2));
FUNC_GROUP_DECL(SALT9, Y22);

#define AA22 211
#define AA22_DESC	SIG_DESC_SET(SCUA4, 27)
SIG_EXPR_DECL_SINGLE(VPOR5, VPO, AA22_DESC, VPO_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOR5, VPOOFF1, AA22_DESC, VPOOFF1_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOR5, VPOOFF2, AA22_DESC, VPOOFF2_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_LIST_DECL(VPOR5, VPO,
		SIG_EXPR_PTR(VPOR5, VPO),
		SIG_EXPR_PTR(VPOR5, VPOOFF1),
		SIG_EXPR_PTR(VPOR5, VPOOFF2));
SIG_EXPR_LIST_ALIAS(AA22, VPOR5, VPO);
SIG_EXPR_LIST_DECL_SINGLE(AA22, SALT10, SALT10, AA22_DESC);
SIG_EXPR_LIST_DECL_SINGLE(AA22, NORD3, PNOR, PNOR_DESC);
SIG_EXPR_LIST_DECL_SINGLE(AA22, GPIOAA3, GPIOAA3);
PIN_DECL_(AA22, SIG_EXPR_LIST_PTR(AA22, VPOR5),
	  SIG_EXPR_LIST_PTR(AA22, SALT10), SIG_EXPR_LIST_PTR(AA22, NORD3),
	  SIG_EXPR_LIST_PTR(AA22, GPIOAA3));
FUNC_GROUP_DECL(SALT10, AA22);

#define U22 212
#define U22_DESC	SIG_DESC_SET(SCUA4, 28)
SIG_EXPR_DECL_SINGLE(VPOR6, VPO, U22_DESC, VPO_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOR6, VPOOFF1, U22_DESC, VPOOFF1_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOR6, VPOOFF2, U22_DESC, VPOOFF2_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_LIST_DECL(VPOR6, VPO,
		SIG_EXPR_PTR(VPOR6, VPO),
		SIG_EXPR_PTR(VPOR6, VPOOFF1),
		SIG_EXPR_PTR(VPOR6, VPOOFF2));
SIG_EXPR_LIST_ALIAS(U22, VPOR6, VPO);
SIG_EXPR_LIST_DECL_SINGLE(U22, SALT11, SALT11, U22_DESC);
SIG_EXPR_LIST_DECL_SINGLE(U22, NORD4, PNOR, PNOR_DESC);
SIG_EXPR_LIST_DECL_SINGLE(U22, GPIOAA4, GPIOAA4);
PIN_DECL_(U22, SIG_EXPR_LIST_PTR(U22, VPOR6), SIG_EXPR_LIST_PTR(U22, SALT11),
		SIG_EXPR_LIST_PTR(U22, NORD4), SIG_EXPR_LIST_PTR(U22, GPIOAA4));
FUNC_GROUP_DECL(SALT11, U22);

#define T20 213
#define T20_DESC	SIG_DESC_SET(SCUA4, 29)
SIG_EXPR_DECL_SINGLE(VPOR7, VPO, T20_DESC, VPO_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOR7, VPOOFF1, T20_DESC, VPOOFF1_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOR7, VPOOFF2, T20_DESC, VPOOFF2_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_LIST_DECL(VPOR7, VPO,
		SIG_EXPR_PTR(VPOR7, VPO),
		SIG_EXPR_PTR(VPOR7, VPOOFF1),
		SIG_EXPR_PTR(VPOR7, VPOOFF2));
SIG_EXPR_LIST_ALIAS(T20, VPOR7, VPO);
SIG_EXPR_LIST_DECL_SINGLE(T20, SALT12, SALT12, T20_DESC);
SIG_EXPR_LIST_DECL_SINGLE(T20, NORD5, PNOR, PNOR_DESC);
SIG_EXPR_LIST_DECL_SINGLE(T20, GPIOAA5, GPIOAA5);
PIN_DECL_(T20, SIG_EXPR_LIST_PTR(T20, VPOR7), SIG_EXPR_LIST_PTR(T20, SALT12),
		SIG_EXPR_LIST_PTR(T20, NORD5), SIG_EXPR_LIST_PTR(T20, GPIOAA5));
FUNC_GROUP_DECL(SALT12, T20);

#define N18 214
#define N18_DESC	SIG_DESC_SET(SCUA4, 30)
SIG_EXPR_DECL_SINGLE(VPOR8, VPO, N18_DESC, VPO_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOR8, VPOOFF1, N18_DESC, VPOOFF1_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOR8, VPOOFF2, N18_DESC, VPOOFF2_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_LIST_DECL(VPOR8, VPO,
		SIG_EXPR_PTR(VPOR8, VPO),
		SIG_EXPR_PTR(VPOR8, VPOOFF1),
		SIG_EXPR_PTR(VPOR8, VPOOFF2));
SIG_EXPR_LIST_ALIAS(N18, VPOR8, VPO);
SIG_EXPR_LIST_DECL_SINGLE(N18, SALT13, SALT13, N18_DESC);
SIG_EXPR_LIST_DECL_SINGLE(N18, NORD6, PNOR, PNOR_DESC);
SIG_EXPR_LIST_DECL_SINGLE(N18, GPIOAA6, GPIOAA6);
PIN_DECL_(N18, SIG_EXPR_LIST_PTR(N18, VPOR8), SIG_EXPR_LIST_PTR(N18, SALT13),
		SIG_EXPR_LIST_PTR(N18, NORD6), SIG_EXPR_LIST_PTR(N18, GPIOAA6));
FUNC_GROUP_DECL(SALT13, N18);

#define P19 215
#define P19_DESC	SIG_DESC_SET(SCUA4, 31)
SIG_EXPR_DECL_SINGLE(VPOR9, VPO, P19_DESC, VPO_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOR9, VPOOFF1, P19_DESC, VPOOFF1_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_DECL_SINGLE(VPOR9, VPOOFF2, P19_DESC, VPOOFF2_DESC, CRT_DVO_ES_DESC);
SIG_EXPR_LIST_DECL(VPOR9, VPO,
		SIG_EXPR_PTR(VPOR9, VPO),
		SIG_EXPR_PTR(VPOR9, VPOOFF1),
		SIG_EXPR_PTR(VPOR9, VPOOFF2));
SIG_EXPR_LIST_ALIAS(P19, VPOR9, VPO);
SIG_EXPR_LIST_DECL_SINGLE(P19, SALT14, SALT14, P19_DESC);
SIG_EXPR_LIST_DECL_SINGLE(P19, NORD7, PNOR, PNOR_DESC);
SIG_EXPR_LIST_DECL_SINGLE(P19, GPIOAA7, GPIOAA7);
PIN_DECL_(P19, SIG_EXPR_LIST_PTR(P19, VPOR9), SIG_EXPR_LIST_PTR(P19, SALT14),
		SIG_EXPR_LIST_PTR(P19, NORD7), SIG_EXPR_LIST_PTR(P19, GPIOAA7));
FUNC_GROUP_DECL(SALT14, P19);

#define N19 216
#define N19_DESC	SIG_DESC_SET(SCUA8, 0)
SIG_EXPR_DECL_SINGLE(VPODE, VPO, N19_DESC, VPO_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPODE, VPOOFF1, N19_DESC, VPOOFF1_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPODE, VPOOFF2, N19_DESC, VPOOFF2_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_LIST_DECL(VPODE, VPO,
		SIG_EXPR_PTR(VPODE, VPO),
		SIG_EXPR_PTR(VPODE, VPOOFF1),
		SIG_EXPR_PTR(VPODE, VPOOFF2));
SIG_EXPR_LIST_ALIAS(N19, VPODE, VPO);
SIG_EXPR_LIST_DECL_SINGLE(N19, NOROE, PNOR, PNOR_DESC);
PIN_DECL_2(N19, GPIOAB0, VPODE, NOROE);

#define T21 217
#define T21_DESC	SIG_DESC_SET(SCUA8, 1)
SIG_EXPR_DECL_SINGLE(VPOHS, VPO, T21_DESC, VPO_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOHS, VPOOFF1, T21_DESC, VPOOFF1_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOHS, VPOOFF2, T21_DESC, VPOOFF2_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_LIST_DECL(VPOHS, VPO,
		SIG_EXPR_PTR(VPOHS, VPO),
		SIG_EXPR_PTR(VPOHS, VPOOFF1),
		SIG_EXPR_PTR(VPOHS, VPOOFF2));
SIG_EXPR_LIST_ALIAS(T21, VPOHS, VPO);
SIG_EXPR_LIST_DECL_SINGLE(T21, NORWE, PNOR, PNOR_DESC);
PIN_DECL_2(T21, GPIOAB1, VPOHS, NORWE);

FUNC_GROUP_DECL(PNOR, Y20, AB20, AB21, AA21, U21, W22, V22, W21, Y21, V21, Y22,
		AA22, U22, T20, N18, P19, N19, T21);

#define T22 218
#define T22_DESC	SIG_DESC_SET(SCUA8, 2)
SIG_EXPR_DECL_SINGLE(VPOVS, VPO, T22_DESC, VPO_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOVS, VPOOFF1, T22_DESC, VPOOFF1_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOVS, VPOOFF2, T22_DESC, VPOOFF2_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_LIST_DECL(VPOVS, VPO,
		SIG_EXPR_PTR(VPOVS, VPO),
		SIG_EXPR_PTR(VPOVS, VPOOFF1),
		SIG_EXPR_PTR(VPOVS, VPOOFF2));
SIG_EXPR_LIST_ALIAS(T22, VPOVS, VPO);
SIG_EXPR_LIST_DECL_SINGLE(T22, WDTRST1, WDTRST1, T22_DESC);
PIN_DECL_2(T22, GPIOAB2, VPOVS, WDTRST1);
FUNC_GROUP_DECL(WDTRST1, T22);

#define R20 219
#define R20_DESC	SIG_DESC_SET(SCUA8, 3)
SIG_EXPR_DECL_SINGLE(VPOCLK, VPO, R20_DESC, VPO_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOCLK, VPOOFF1, R20_DESC, VPOOFF1_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_DECL_SINGLE(VPOCLK, VPOOFF2, R20_DESC, VPOOFF2_DESC, CRT_DVO_EN_DESC);
SIG_EXPR_LIST_DECL(VPOCLK, VPO,
		SIG_EXPR_PTR(VPOCLK, VPO),
		SIG_EXPR_PTR(VPOCLK, VPOOFF1),
		SIG_EXPR_PTR(VPOCLK, VPOOFF2));
SIG_EXPR_LIST_ALIAS(R20, VPOCLK, VPO);
SIG_EXPR_LIST_DECL_SINGLE(R20, WDTRST2, WDTRST2, R20_DESC);
PIN_DECL_2(R20, GPIOAB3, VPOCLK, WDTRST2);
FUNC_GROUP_DECL(WDTRST2, R20);

FUNC_GROUP_DECL(VPO, V20, U19, R18, P18, R19, W20, U20, AA20, Y20, AB20,
		AB21, AA21, U21, W22, V22, W21, Y21, V21, Y22, AA22, U22, T20,
		N18, P19, N19, T21, T22, R20);

#define ESPI_DESC	SIG_DESC_SET(HW_STRAP1, 25)

#define G21 224
SIG_EXPR_LIST_DECL_SINGLE(G21, ESPID0, ESPI, ESPI_DESC);
SIG_EXPR_LIST_DECL_SINGLE(G21, LAD0, LAD0, SIG_DESC_SET(SCUAC, 0));
PIN_DECL_2(G21, GPIOAC0, ESPID0, LAD0);
FUNC_GROUP_DECL(LAD0, G21);

#define G20 225
SIG_EXPR_LIST_DECL_SINGLE(G20, ESPID1, ESPI, ESPI_DESC);
SIG_EXPR_LIST_DECL_SINGLE(G20, LAD1, LAD1, SIG_DESC_SET(SCUAC, 1));
PIN_DECL_2(G20, GPIOAC1, ESPID1, LAD1);
FUNC_GROUP_DECL(LAD1, G20);

#define D22 226
SIG_EXPR_LIST_DECL_SINGLE(D22, ESPID2, ESPI, ESPI_DESC);
SIG_EXPR_LIST_DECL_SINGLE(D22, LAD2, LAD2, SIG_DESC_SET(SCUAC, 2));
PIN_DECL_2(D22, GPIOAC2, ESPID2, LAD2);
FUNC_GROUP_DECL(LAD2, D22);

#define E22 227
SIG_EXPR_LIST_DECL_SINGLE(E22, ESPID3, ESPI, ESPI_DESC);
SIG_EXPR_LIST_DECL_SINGLE(E22, LAD3, LAD3, SIG_DESC_SET(SCUAC, 3));
PIN_DECL_2(E22, GPIOAC3, ESPID3, LAD3);
FUNC_GROUP_DECL(LAD3, E22);

#define C22 228
SIG_EXPR_LIST_DECL_SINGLE(C22, ESPICK, ESPI, ESPI_DESC);
SIG_EXPR_LIST_DECL_SINGLE(C22, LCLK, LCLK, SIG_DESC_SET(SCUAC, 4));
PIN_DECL_2(C22, GPIOAC4, ESPICK, LCLK);
FUNC_GROUP_DECL(LCLK, C22);

#define F21 229
SIG_EXPR_LIST_DECL_SINGLE(F21, ESPICS, ESPI, ESPI_DESC);
SIG_EXPR_LIST_DECL_SINGLE(F21, LFRAME, LFRAME, SIG_DESC_SET(SCUAC, 5));
PIN_DECL_2(F21, GPIOAC5, ESPICS, LFRAME);
FUNC_GROUP_DECL(LFRAME, F21);

#define F22 230
SIG_EXPR_LIST_DECL_SINGLE(F22, ESPIALT, ESPI, ESPI_DESC);
SIG_EXPR_LIST_DECL_SINGLE(F22, LSIRQ, LSIRQ, SIG_DESC_SET(SCUAC, 6));
PIN_DECL_2(F22, GPIOAC6, ESPIALT, LSIRQ);
FUNC_GROUP_DECL(LSIRQ, F22);

#define G22 231
SIG_EXPR_LIST_DECL_SINGLE(G22, ESPIRST, ESPI, ESPI_DESC);
SIG_EXPR_LIST_DECL_SINGLE(G22, LPCRST, LPCRST, SIG_DESC_SET(SCUAC, 7));
PIN_DECL_2(G22, GPIOAC7, ESPIRST, LPCRST);
FUNC_GROUP_DECL(LPCRST, G22);

FUNC_GROUP_DECL(ESPI, G21, G20, D22, E22, C22, F21, F22, G22);

#define A7 232
SIG_EXPR_LIST_DECL_SINGLE(A7, USB2AHDP, USB2AH, SIG_DESC_SET(SCU90, 29));
SIG_EXPR_LIST_DECL_SINGLE(A7, USB2ADDP, USB2AD, SIG_DESC_BIT(SCU90, 29, 0));
PIN_DECL_(A7, SIG_EXPR_LIST_PTR(A7, USB2AHDP), SIG_EXPR_LIST_PTR(A7, USB2ADDP));

#define A8 233
SIG_EXPR_LIST_DECL_SINGLE(A8, USB2AHDN, USB2AH, SIG_DESC_SET(SCU90, 29));
SIG_EXPR_LIST_DECL_SINGLE(A8, USB2ADDN, USB2AD, SIG_DESC_BIT(SCU90, 29, 0));
PIN_DECL_(A8, SIG_EXPR_LIST_PTR(A8, USB2AHDN), SIG_EXPR_LIST_PTR(A8, USB2ADDN));

FUNC_GROUP_DECL(USB2AH, A7, A8);
FUNC_GROUP_DECL(USB2AD, A7, A8);

#define USB11BHID_DESC  { ASPEED_IP_SCU, SCU94, GENMASK(14, 13), 0, 0 }
#define USB2BD_DESC   { ASPEED_IP_SCU, SCU94, GENMASK(14, 13), 1, 0 }
#define USB2BH1_DESC { ASPEED_IP_SCU, SCU94, GENMASK(14, 13), 2, 0 }
#define USB2BH2_DESC { ASPEED_IP_SCU, SCU94, GENMASK(14, 13), 3, 0 }

#define B6 234
SIG_EXPR_LIST_DECL_SINGLE(B6, USB11BDP, USB11BHID, USB11BHID_DESC);
SIG_EXPR_LIST_DECL_SINGLE(B6, USB2BDDP, USB2BD, USB2BD_DESC);
SIG_EXPR_DECL_SINGLE(USB2BHDP1, USB2BH, USB2BH1_DESC);
SIG_EXPR_DECL_SINGLE(USB2BHDP2, USB2BH, USB2BH2_DESC);
SIG_EXPR_LIST_DECL(USB2BHDP, USB2BH,
		SIG_EXPR_PTR(USB2BHDP1, USB2BH),
		SIG_EXPR_PTR(USB2BHDP2, USB2BH));
SIG_EXPR_LIST_ALIAS(B6, USB2BHDP, USB2BH);
PIN_DECL_(B6, SIG_EXPR_LIST_PTR(B6, USB11BDP), SIG_EXPR_LIST_PTR(B6, USB2BDDP),
		SIG_EXPR_LIST_PTR(B6, USB2BHDP));

#define A6 235
SIG_EXPR_LIST_DECL_SINGLE(A6, USB11BDN, USB11BHID, USB11BHID_DESC);
SIG_EXPR_LIST_DECL_SINGLE(A6, USB2BDN, USB2BD, USB2BD_DESC);
SIG_EXPR_DECL_SINGLE(USB2BHDN1, USB2BH, USB2BH1_DESC);
SIG_EXPR_DECL_SINGLE(USB2BHDN2, USB2BH, USB2BH2_DESC);
SIG_EXPR_LIST_DECL(USB2BHDN, USB2BH,
		SIG_EXPR_PTR(USB2BHDN1, USB2BH),
		SIG_EXPR_PTR(USB2BHDN2, USB2BH));
SIG_EXPR_LIST_ALIAS(A6, USB2BHDN, USB2BH);
PIN_DECL_(A6, SIG_EXPR_LIST_PTR(A6, USB11BDN), SIG_EXPR_LIST_PTR(A6, USB2BDN),
		SIG_EXPR_LIST_PTR(A6, USB2BHDN));

FUNC_GROUP_DECL(USB11BHID, B6, A6);
FUNC_GROUP_DECL(USB2BD, B6, A6);
FUNC_GROUP_DECL(USB2BH, B6, A6);

/* Pins, groups and functions are sort(1):ed alphabetically for sanity */

static struct pinctrl_pin_desc aspeed_g5_pins[ASPEED_G5_NR_PINS] = {
	ASPEED_PINCTRL_PIN(A10),
	ASPEED_PINCTRL_PIN(A11),
	ASPEED_PINCTRL_PIN(A12),
	ASPEED_PINCTRL_PIN(A13),
	ASPEED_PINCTRL_PIN(A14),
	ASPEED_PINCTRL_PIN(A15),
	ASPEED_PINCTRL_PIN(A16),
	ASPEED_PINCTRL_PIN(A17),
	ASPEED_PINCTRL_PIN(A18),
	ASPEED_PINCTRL_PIN(A19),
	ASPEED_PINCTRL_PIN(A2),
	ASPEED_PINCTRL_PIN(A20),
	ASPEED_PINCTRL_PIN(A21),
	ASPEED_PINCTRL_PIN(A3),
	ASPEED_PINCTRL_PIN(A4),
	ASPEED_PINCTRL_PIN(A5),
	ASPEED_PINCTRL_PIN(A6),
	ASPEED_PINCTRL_PIN(A7),
	ASPEED_PINCTRL_PIN(A8),
	ASPEED_PINCTRL_PIN(A9),
	ASPEED_PINCTRL_PIN(AA1),
	ASPEED_PINCTRL_PIN(AA19),
	ASPEED_PINCTRL_PIN(AA2),
	ASPEED_PINCTRL_PIN(AA20),
	ASPEED_PINCTRL_PIN(AA21),
	ASPEED_PINCTRL_PIN(AA22),
	ASPEED_PINCTRL_PIN(AA3),
	ASPEED_PINCTRL_PIN(AA4),
	ASPEED_PINCTRL_PIN(AA5),
	ASPEED_PINCTRL_PIN(AB2),
	ASPEED_PINCTRL_PIN(AB20),
	ASPEED_PINCTRL_PIN(AB21),
	ASPEED_PINCTRL_PIN(AB3),
	ASPEED_PINCTRL_PIN(AB4),
	ASPEED_PINCTRL_PIN(AB5),
	ASPEED_PINCTRL_PIN(B1),
	ASPEED_PINCTRL_PIN(B10),
	ASPEED_PINCTRL_PIN(B11),
	ASPEED_PINCTRL_PIN(B12),
	ASPEED_PINCTRL_PIN(B13),
	ASPEED_PINCTRL_PIN(B14),
	ASPEED_PINCTRL_PIN(B15),
	ASPEED_PINCTRL_PIN(B16),
	ASPEED_PINCTRL_PIN(B17),
	ASPEED_PINCTRL_PIN(B18),
	ASPEED_PINCTRL_PIN(B19),
	ASPEED_PINCTRL_PIN(B2),
	ASPEED_PINCTRL_PIN(B20),
	ASPEED_PINCTRL_PIN(B21),
	ASPEED_PINCTRL_PIN(B22),
	ASPEED_PINCTRL_PIN(B3),
	ASPEED_PINCTRL_PIN(B4),
	ASPEED_PINCTRL_PIN(B5),
	ASPEED_PINCTRL_PIN(B6),
	ASPEED_PINCTRL_PIN(B9),
	ASPEED_PINCTRL_PIN(C1),
	ASPEED_PINCTRL_PIN(C11),
	ASPEED_PINCTRL_PIN(C12),
	ASPEED_PINCTRL_PIN(C13),
	ASPEED_PINCTRL_PIN(C14),
	ASPEED_PINCTRL_PIN(C15),
	ASPEED_PINCTRL_PIN(C16),
	ASPEED_PINCTRL_PIN(C17),
	ASPEED_PINCTRL_PIN(C18),
	ASPEED_PINCTRL_PIN(C19),
	ASPEED_PINCTRL_PIN(C2),
	ASPEED_PINCTRL_PIN(C20),
	ASPEED_PINCTRL_PIN(C21),
	ASPEED_PINCTRL_PIN(C22),
	ASPEED_PINCTRL_PIN(C3),
	ASPEED_PINCTRL_PIN(C4),
	ASPEED_PINCTRL_PIN(C5),
	ASPEED_PINCTRL_PIN(D1),
	ASPEED_PINCTRL_PIN(D10),
	ASPEED_PINCTRL_PIN(D13),
	ASPEED_PINCTRL_PIN(D14),
	ASPEED_PINCTRL_PIN(D15),
	ASPEED_PINCTRL_PIN(D16),
	ASPEED_PINCTRL_PIN(D17),
	ASPEED_PINCTRL_PIN(D18),
	ASPEED_PINCTRL_PIN(D19),
	ASPEED_PINCTRL_PIN(D2),
	ASPEED_PINCTRL_PIN(D20),
	ASPEED_PINCTRL_PIN(D21),
	ASPEED_PINCTRL_PIN(D22),
	ASPEED_PINCTRL_PIN(D4),
	ASPEED_PINCTRL_PIN(D5),
	ASPEED_PINCTRL_PIN(D6),
	ASPEED_PINCTRL_PIN(D7),
	ASPEED_PINCTRL_PIN(D8),
	ASPEED_PINCTRL_PIN(D9),
	ASPEED_PINCTRL_PIN(E1),
	ASPEED_PINCTRL_PIN(E10),
	ASPEED_PINCTRL_PIN(E12),
	ASPEED_PINCTRL_PIN(E13),
	ASPEED_PINCTRL_PIN(E14),
	ASPEED_PINCTRL_PIN(E15),
	ASPEED_PINCTRL_PIN(E16),
	ASPEED_PINCTRL_PIN(E17),
	ASPEED_PINCTRL_PIN(E18),
	ASPEED_PINCTRL_PIN(E19),
	ASPEED_PINCTRL_PIN(E2),
	ASPEED_PINCTRL_PIN(E20),
	ASPEED_PINCTRL_PIN(E21),
	ASPEED_PINCTRL_PIN(E22),
	ASPEED_PINCTRL_PIN(E3),
	ASPEED_PINCTRL_PIN(E6),
	ASPEED_PINCTRL_PIN(E7),
	ASPEED_PINCTRL_PIN(E9),
	ASPEED_PINCTRL_PIN(F1),
	ASPEED_PINCTRL_PIN(F17),
	ASPEED_PINCTRL_PIN(F18),
	ASPEED_PINCTRL_PIN(F19),
	ASPEED_PINCTRL_PIN(F2),
	ASPEED_PINCTRL_PIN(F20),
	ASPEED_PINCTRL_PIN(F21),
	ASPEED_PINCTRL_PIN(F22),
	ASPEED_PINCTRL_PIN(F3),
	ASPEED_PINCTRL_PIN(F4),
	ASPEED_PINCTRL_PIN(F5),
	ASPEED_PINCTRL_PIN(F9),
	ASPEED_PINCTRL_PIN(G1),
	ASPEED_PINCTRL_PIN(G17),
	ASPEED_PINCTRL_PIN(G18),
	ASPEED_PINCTRL_PIN(G2),
	ASPEED_PINCTRL_PIN(G20),
	ASPEED_PINCTRL_PIN(G21),
	ASPEED_PINCTRL_PIN(G22),
	ASPEED_PINCTRL_PIN(G3),
	ASPEED_PINCTRL_PIN(G4),
	ASPEED_PINCTRL_PIN(G5),
	ASPEED_PINCTRL_PIN(H18),
	ASPEED_PINCTRL_PIN(H19),
	ASPEED_PINCTRL_PIN(H20),
	ASPEED_PINCTRL_PIN(H21),
	ASPEED_PINCTRL_PIN(H22),
	ASPEED_PINCTRL_PIN(H3),
	ASPEED_PINCTRL_PIN(H4),
	ASPEED_PINCTRL_PIN(H5),
	ASPEED_PINCTRL_PIN(J18),
	ASPEED_PINCTRL_PIN(J19),
	ASPEED_PINCTRL_PIN(J20),
	ASPEED_PINCTRL_PIN(K18),
	ASPEED_PINCTRL_PIN(K19),
	ASPEED_PINCTRL_PIN(L1),
	ASPEED_PINCTRL_PIN(L18),
	ASPEED_PINCTRL_PIN(L19),
	ASPEED_PINCTRL_PIN(L2),
	ASPEED_PINCTRL_PIN(L3),
	ASPEED_PINCTRL_PIN(L4),
	ASPEED_PINCTRL_PIN(M18),
	ASPEED_PINCTRL_PIN(M19),
	ASPEED_PINCTRL_PIN(M20),
	ASPEED_PINCTRL_PIN(N1),
	ASPEED_PINCTRL_PIN(N18),
	ASPEED_PINCTRL_PIN(N19),
	ASPEED_PINCTRL_PIN(N2),
	ASPEED_PINCTRL_PIN(N20),
	ASPEED_PINCTRL_PIN(N21),
	ASPEED_PINCTRL_PIN(N22),
	ASPEED_PINCTRL_PIN(N3),
	ASPEED_PINCTRL_PIN(N4),
	ASPEED_PINCTRL_PIN(N5),
	ASPEED_PINCTRL_PIN(P1),
	ASPEED_PINCTRL_PIN(P18),
	ASPEED_PINCTRL_PIN(P19),
	ASPEED_PINCTRL_PIN(P2),
	ASPEED_PINCTRL_PIN(P20),
	ASPEED_PINCTRL_PIN(P21),
	ASPEED_PINCTRL_PIN(P22),
	ASPEED_PINCTRL_PIN(P3),
	ASPEED_PINCTRL_PIN(P4),
	ASPEED_PINCTRL_PIN(P5),
	ASPEED_PINCTRL_PIN(R1),
	ASPEED_PINCTRL_PIN(R18),
	ASPEED_PINCTRL_PIN(R19),
	ASPEED_PINCTRL_PIN(R2),
	ASPEED_PINCTRL_PIN(R20),
	ASPEED_PINCTRL_PIN(R21),
	ASPEED_PINCTRL_PIN(R22),
	ASPEED_PINCTRL_PIN(R3),
	ASPEED_PINCTRL_PIN(R4),
	ASPEED_PINCTRL_PIN(R5),
	ASPEED_PINCTRL_PIN(T1),
	ASPEED_PINCTRL_PIN(T17),
	ASPEED_PINCTRL_PIN(T19),
	ASPEED_PINCTRL_PIN(T2),
	ASPEED_PINCTRL_PIN(T20),
	ASPEED_PINCTRL_PIN(T21),
	ASPEED_PINCTRL_PIN(T22),
	ASPEED_PINCTRL_PIN(T3),
	ASPEED_PINCTRL_PIN(T4),
	ASPEED_PINCTRL_PIN(T5),
	ASPEED_PINCTRL_PIN(U1),
	ASPEED_PINCTRL_PIN(U19),
	ASPEED_PINCTRL_PIN(U2),
	ASPEED_PINCTRL_PIN(U20),
	ASPEED_PINCTRL_PIN(U21),
	ASPEED_PINCTRL_PIN(U22),
	ASPEED_PINCTRL_PIN(U3),
	ASPEED_PINCTRL_PIN(U4),
	ASPEED_PINCTRL_PIN(U5),
	ASPEED_PINCTRL_PIN(V1),
	ASPEED_PINCTRL_PIN(V19),
	ASPEED_PINCTRL_PIN(V2),
	ASPEED_PINCTRL_PIN(V20),
	ASPEED_PINCTRL_PIN(V21),
	ASPEED_PINCTRL_PIN(V22),
	ASPEED_PINCTRL_PIN(V3),
	ASPEED_PINCTRL_PIN(V4),
	ASPEED_PINCTRL_PIN(V5),
	ASPEED_PINCTRL_PIN(V6),
	ASPEED_PINCTRL_PIN(W1),
	ASPEED_PINCTRL_PIN(W19),
	ASPEED_PINCTRL_PIN(W2),
	ASPEED_PINCTRL_PIN(W20),
	ASPEED_PINCTRL_PIN(W21),
	ASPEED_PINCTRL_PIN(W22),
	ASPEED_PINCTRL_PIN(W3),
	ASPEED_PINCTRL_PIN(W4),
	ASPEED_PINCTRL_PIN(W5),
	ASPEED_PINCTRL_PIN(W6),
	ASPEED_PINCTRL_PIN(Y1),
	ASPEED_PINCTRL_PIN(Y19),
	ASPEED_PINCTRL_PIN(Y2),
	ASPEED_PINCTRL_PIN(Y20),
	ASPEED_PINCTRL_PIN(Y21),
	ASPEED_PINCTRL_PIN(Y22),
	ASPEED_PINCTRL_PIN(Y3),
	ASPEED_PINCTRL_PIN(Y4),
	ASPEED_PINCTRL_PIN(Y5),
	ASPEED_PINCTRL_PIN(Y6),
};

static const struct aspeed_pin_group aspeed_g5_groups[] = {
	ASPEED_PINCTRL_GROUP(ACPI),
	ASPEED_PINCTRL_GROUP(ADC0),
	ASPEED_PINCTRL_GROUP(ADC1),
	ASPEED_PINCTRL_GROUP(ADC10),
	ASPEED_PINCTRL_GROUP(ADC11),
	ASPEED_PINCTRL_GROUP(ADC12),
	ASPEED_PINCTRL_GROUP(ADC13),
	ASPEED_PINCTRL_GROUP(ADC14),
	ASPEED_PINCTRL_GROUP(ADC15),
	ASPEED_PINCTRL_GROUP(ADC2),
	ASPEED_PINCTRL_GROUP(ADC3),
	ASPEED_PINCTRL_GROUP(ADC4),
	ASPEED_PINCTRL_GROUP(ADC5),
	ASPEED_PINCTRL_GROUP(ADC6),
	ASPEED_PINCTRL_GROUP(ADC7),
	ASPEED_PINCTRL_GROUP(ADC8),
	ASPEED_PINCTRL_GROUP(ADC9),
	ASPEED_PINCTRL_GROUP(BMCINT),
	ASPEED_PINCTRL_GROUP(DDCCLK),
	ASPEED_PINCTRL_GROUP(DDCDAT),
	ASPEED_PINCTRL_GROUP(ESPI),
	ASPEED_PINCTRL_GROUP(FWSPICS1),
	ASPEED_PINCTRL_GROUP(FWSPICS2),
	ASPEED_PINCTRL_GROUP(GPID0),
	ASPEED_PINCTRL_GROUP(GPID2),
	ASPEED_PINCTRL_GROUP(GPID4),
	ASPEED_PINCTRL_GROUP(GPID6),
	ASPEED_PINCTRL_GROUP(GPIE0),
	ASPEED_PINCTRL_GROUP(GPIE2),
	ASPEED_PINCTRL_GROUP(GPIE4),
	ASPEED_PINCTRL_GROUP(GPIE6),
	ASPEED_PINCTRL_GROUP(I2C10),
	ASPEED_PINCTRL_GROUP(I2C11),
	ASPEED_PINCTRL_GROUP(I2C12),
	ASPEED_PINCTRL_GROUP(I2C13),
	ASPEED_PINCTRL_GROUP(I2C14),
	ASPEED_PINCTRL_GROUP(I2C3),
	ASPEED_PINCTRL_GROUP(I2C4),
	ASPEED_PINCTRL_GROUP(I2C5),
	ASPEED_PINCTRL_GROUP(I2C6),
	ASPEED_PINCTRL_GROUP(I2C7),
	ASPEED_PINCTRL_GROUP(I2C8),
	ASPEED_PINCTRL_GROUP(I2C9),
	ASPEED_PINCTRL_GROUP(LAD0),
	ASPEED_PINCTRL_GROUP(LAD1),
	ASPEED_PINCTRL_GROUP(LAD2),
	ASPEED_PINCTRL_GROUP(LAD3),
	ASPEED_PINCTRL_GROUP(LCLK),
	ASPEED_PINCTRL_GROUP(LFRAME),
	ASPEED_PINCTRL_GROUP(LPCHC),
	ASPEED_PINCTRL_GROUP(LPCPD),
	ASPEED_PINCTRL_GROUP(LPCPLUS),
	ASPEED_PINCTRL_GROUP(LPCPME),
	ASPEED_PINCTRL_GROUP(LPCRST),
	ASPEED_PINCTRL_GROUP(LPCSMI),
	ASPEED_PINCTRL_GROUP(LSIRQ),
	ASPEED_PINCTRL_GROUP(MAC1LINK),
	ASPEED_PINCTRL_GROUP(MAC2LINK),
	ASPEED_PINCTRL_GROUP(MDIO1),
	ASPEED_PINCTRL_GROUP(MDIO2),
	ASPEED_PINCTRL_GROUP(NCTS1),
	ASPEED_PINCTRL_GROUP(NCTS2),
	ASPEED_PINCTRL_GROUP(NCTS3),
	ASPEED_PINCTRL_GROUP(NCTS4),
	ASPEED_PINCTRL_GROUP(NDCD1),
	ASPEED_PINCTRL_GROUP(NDCD2),
	ASPEED_PINCTRL_GROUP(NDCD3),
	ASPEED_PINCTRL_GROUP(NDCD4),
	ASPEED_PINCTRL_GROUP(NDSR1),
	ASPEED_PINCTRL_GROUP(NDSR2),
	ASPEED_PINCTRL_GROUP(NDSR3),
	ASPEED_PINCTRL_GROUP(NDSR4),
	ASPEED_PINCTRL_GROUP(NDTR1),
	ASPEED_PINCTRL_GROUP(NDTR2),
	ASPEED_PINCTRL_GROUP(NDTR3),
	ASPEED_PINCTRL_GROUP(NDTR4),
	ASPEED_PINCTRL_GROUP(NRI1),
	ASPEED_PINCTRL_GROUP(NRI2),
	ASPEED_PINCTRL_GROUP(NRI3),
	ASPEED_PINCTRL_GROUP(NRI4),
	ASPEED_PINCTRL_GROUP(NRTS1),
	ASPEED_PINCTRL_GROUP(NRTS2),
	ASPEED_PINCTRL_GROUP(NRTS3),
	ASPEED_PINCTRL_GROUP(NRTS4),
	ASPEED_PINCTRL_GROUP(OSCCLK),
	ASPEED_PINCTRL_GROUP(PEWAKE),
	ASPEED_PINCTRL_GROUP(PNOR),
	ASPEED_PINCTRL_GROUP(PWM0),
	ASPEED_PINCTRL_GROUP(PWM1),
	ASPEED_PINCTRL_GROUP(PWM2),
	ASPEED_PINCTRL_GROUP(PWM3),
	ASPEED_PINCTRL_GROUP(PWM4),
	ASPEED_PINCTRL_GROUP(PWM5),
	ASPEED_PINCTRL_GROUP(PWM6),
	ASPEED_PINCTRL_GROUP(PWM7),
	ASPEED_PINCTRL_GROUP(RGMII1),
	ASPEED_PINCTRL_GROUP(RGMII2),
	ASPEED_PINCTRL_GROUP(RMII1),
	ASPEED_PINCTRL_GROUP(RMII2),
	ASPEED_PINCTRL_GROUP(RXD1),
	ASPEED_PINCTRL_GROUP(RXD2),
	ASPEED_PINCTRL_GROUP(RXD3),
	ASPEED_PINCTRL_GROUP(RXD4),
	ASPEED_PINCTRL_GROUP(SALT1),
	ASPEED_PINCTRL_GROUP(SALT10),
	ASPEED_PINCTRL_GROUP(SALT11),
	ASPEED_PINCTRL_GROUP(SALT12),
	ASPEED_PINCTRL_GROUP(SALT13),
	ASPEED_PINCTRL_GROUP(SALT14),
	ASPEED_PINCTRL_GROUP(SALT2),
	ASPEED_PINCTRL_GROUP(SALT3),
	ASPEED_PINCTRL_GROUP(SALT4),
	ASPEED_PINCTRL_GROUP(SALT5),
	ASPEED_PINCTRL_GROUP(SALT6),
	ASPEED_PINCTRL_GROUP(SALT7),
	ASPEED_PINCTRL_GROUP(SALT8),
	ASPEED_PINCTRL_GROUP(SALT9),
	ASPEED_PINCTRL_GROUP(SCL1),
	ASPEED_PINCTRL_GROUP(SCL2),
	ASPEED_PINCTRL_GROUP(SD1),
	ASPEED_PINCTRL_GROUP(SD2),
	ASPEED_PINCTRL_GROUP(SDA1),
	ASPEED_PINCTRL_GROUP(SDA2),
	ASPEED_PINCTRL_GROUP(SGPM),
	ASPEED_PINCTRL_GROUP(SGPS1),
	ASPEED_PINCTRL_GROUP(SGPS2),
	ASPEED_PINCTRL_GROUP(SIOONCTRL),
	ASPEED_PINCTRL_GROUP(SIOPBI),
	ASPEED_PINCTRL_GROUP(SIOPBO),
	ASPEED_PINCTRL_GROUP(SIOPWREQ),
	ASPEED_PINCTRL_GROUP(SIOPWRGD),
	ASPEED_PINCTRL_GROUP(SIOS3),
	ASPEED_PINCTRL_GROUP(SIOS5),
	ASPEED_PINCTRL_GROUP(SIOSCI),
	ASPEED_PINCTRL_GROUP(SPI1),
	ASPEED_PINCTRL_GROUP(SPI1CS1),
	ASPEED_PINCTRL_GROUP(SPI1DEBUG),
	ASPEED_PINCTRL_GROUP(SPI1PASSTHRU),
	ASPEED_PINCTRL_GROUP(SPI2CK),
	ASPEED_PINCTRL_GROUP(SPI2CS0),
	ASPEED_PINCTRL_GROUP(SPI2CS1),
	ASPEED_PINCTRL_GROUP(SPI2MISO),
	ASPEED_PINCTRL_GROUP(SPI2MOSI),
	ASPEED_PINCTRL_GROUP(TIMER3),
	ASPEED_PINCTRL_GROUP(TIMER4),
	ASPEED_PINCTRL_GROUP(TIMER5),
	ASPEED_PINCTRL_GROUP(TIMER6),
	ASPEED_PINCTRL_GROUP(TIMER7),
	ASPEED_PINCTRL_GROUP(TIMER8),
	ASPEED_PINCTRL_GROUP(TXD1),
	ASPEED_PINCTRL_GROUP(TXD2),
	ASPEED_PINCTRL_GROUP(TXD3),
	ASPEED_PINCTRL_GROUP(TXD4),
	ASPEED_PINCTRL_GROUP(UART6),
	ASPEED_PINCTRL_GROUP(USB11BHID),
	ASPEED_PINCTRL_GROUP(USB2AD),
	ASPEED_PINCTRL_GROUP(USB2AH),
	ASPEED_PINCTRL_GROUP(USB2BD),
	ASPEED_PINCTRL_GROUP(USB2BH),
	ASPEED_PINCTRL_GROUP(USBCKI),
	ASPEED_PINCTRL_GROUP(VGABIOSROM),
	ASPEED_PINCTRL_GROUP(VGAHS),
	ASPEED_PINCTRL_GROUP(VGAVS),
	ASPEED_PINCTRL_GROUP(VPI24),
	ASPEED_PINCTRL_GROUP(VPO),
	ASPEED_PINCTRL_GROUP(WDTRST1),
	ASPEED_PINCTRL_GROUP(WDTRST2),
};

static const struct aspeed_pin_function aspeed_g5_functions[] = {
	ASPEED_PINCTRL_FUNC(ACPI),
	ASPEED_PINCTRL_FUNC(ADC0),
	ASPEED_PINCTRL_FUNC(ADC1),
	ASPEED_PINCTRL_FUNC(ADC10),
	ASPEED_PINCTRL_FUNC(ADC11),
	ASPEED_PINCTRL_FUNC(ADC12),
	ASPEED_PINCTRL_FUNC(ADC13),
	ASPEED_PINCTRL_FUNC(ADC14),
	ASPEED_PINCTRL_FUNC(ADC15),
	ASPEED_PINCTRL_FUNC(ADC2),
	ASPEED_PINCTRL_FUNC(ADC3),
	ASPEED_PINCTRL_FUNC(ADC4),
	ASPEED_PINCTRL_FUNC(ADC5),
	ASPEED_PINCTRL_FUNC(ADC6),
	ASPEED_PINCTRL_FUNC(ADC7),
	ASPEED_PINCTRL_FUNC(ADC8),
	ASPEED_PINCTRL_FUNC(ADC9),
	ASPEED_PINCTRL_FUNC(BMCINT),
	ASPEED_PINCTRL_FUNC(DDCCLK),
	ASPEED_PINCTRL_FUNC(DDCDAT),
	ASPEED_PINCTRL_FUNC(ESPI),
	ASPEED_PINCTRL_FUNC(FWSPICS1),
	ASPEED_PINCTRL_FUNC(FWSPICS2),
	ASPEED_PINCTRL_FUNC(GPID0),
	ASPEED_PINCTRL_FUNC(GPID2),
	ASPEED_PINCTRL_FUNC(GPID4),
	ASPEED_PINCTRL_FUNC(GPID6),
	ASPEED_PINCTRL_FUNC(GPIE0),
	ASPEED_PINCTRL_FUNC(GPIE2),
	ASPEED_PINCTRL_FUNC(GPIE4),
	ASPEED_PINCTRL_FUNC(GPIE6),
	ASPEED_PINCTRL_FUNC(I2C10),
	ASPEED_PINCTRL_FUNC(I2C11),
	ASPEED_PINCTRL_FUNC(I2C12),
	ASPEED_PINCTRL_FUNC(I2C13),
	ASPEED_PINCTRL_FUNC(I2C14),
	ASPEED_PINCTRL_FUNC(I2C3),
	ASPEED_PINCTRL_FUNC(I2C4),
	ASPEED_PINCTRL_FUNC(I2C5),
	ASPEED_PINCTRL_FUNC(I2C6),
	ASPEED_PINCTRL_FUNC(I2C7),
	ASPEED_PINCTRL_FUNC(I2C8),
	ASPEED_PINCTRL_FUNC(I2C9),
	ASPEED_PINCTRL_FUNC(LAD0),
	ASPEED_PINCTRL_FUNC(LAD1),
	ASPEED_PINCTRL_FUNC(LAD2),
	ASPEED_PINCTRL_FUNC(LAD3),
	ASPEED_PINCTRL_FUNC(LCLK),
	ASPEED_PINCTRL_FUNC(LFRAME),
	ASPEED_PINCTRL_FUNC(LPCHC),
	ASPEED_PINCTRL_FUNC(LPCPD),
	ASPEED_PINCTRL_FUNC(LPCPLUS),
	ASPEED_PINCTRL_FUNC(LPCPME),
	ASPEED_PINCTRL_FUNC(LPCRST),
	ASPEED_PINCTRL_FUNC(LPCSMI),
	ASPEED_PINCTRL_FUNC(LSIRQ),
	ASPEED_PINCTRL_FUNC(MAC1LINK),
	ASPEED_PINCTRL_FUNC(MAC2LINK),
	ASPEED_PINCTRL_FUNC(MDIO1),
	ASPEED_PINCTRL_FUNC(MDIO2),
	ASPEED_PINCTRL_FUNC(NCTS1),
	ASPEED_PINCTRL_FUNC(NCTS2),
	ASPEED_PINCTRL_FUNC(NCTS3),
	ASPEED_PINCTRL_FUNC(NCTS4),
	ASPEED_PINCTRL_FUNC(NDCD1),
	ASPEED_PINCTRL_FUNC(NDCD2),
	ASPEED_PINCTRL_FUNC(NDCD3),
	ASPEED_PINCTRL_FUNC(NDCD4),
	ASPEED_PINCTRL_FUNC(NDSR1),
	ASPEED_PINCTRL_FUNC(NDSR2),
	ASPEED_PINCTRL_FUNC(NDSR3),
	ASPEED_PINCTRL_FUNC(NDSR4),
	ASPEED_PINCTRL_FUNC(NDTR1),
	ASPEED_PINCTRL_FUNC(NDTR2),
	ASPEED_PINCTRL_FUNC(NDTR3),
	ASPEED_PINCTRL_FUNC(NDTR4),
	ASPEED_PINCTRL_FUNC(NRI1),
	ASPEED_PINCTRL_FUNC(NRI2),
	ASPEED_PINCTRL_FUNC(NRI3),
	ASPEED_PINCTRL_FUNC(NRI4),
	ASPEED_PINCTRL_FUNC(NRTS1),
	ASPEED_PINCTRL_FUNC(NRTS2),
	ASPEED_PINCTRL_FUNC(NRTS3),
	ASPEED_PINCTRL_FUNC(NRTS4),
	ASPEED_PINCTRL_FUNC(OSCCLK),
	ASPEED_PINCTRL_FUNC(PEWAKE),
	ASPEED_PINCTRL_FUNC(PNOR),
	ASPEED_PINCTRL_FUNC(PWM0),
	ASPEED_PINCTRL_FUNC(PWM1),
	ASPEED_PINCTRL_FUNC(PWM2),
	ASPEED_PINCTRL_FUNC(PWM3),
	ASPEED_PINCTRL_FUNC(PWM4),
	ASPEED_PINCTRL_FUNC(PWM5),
	ASPEED_PINCTRL_FUNC(PWM6),
	ASPEED_PINCTRL_FUNC(PWM7),
	ASPEED_PINCTRL_FUNC(RGMII1),
	ASPEED_PINCTRL_FUNC(RGMII2),
	ASPEED_PINCTRL_FUNC(RMII1),
	ASPEED_PINCTRL_FUNC(RMII2),
	ASPEED_PINCTRL_FUNC(RXD1),
	ASPEED_PINCTRL_FUNC(RXD2),
	ASPEED_PINCTRL_FUNC(RXD3),
	ASPEED_PINCTRL_FUNC(RXD4),
	ASPEED_PINCTRL_FUNC(SALT1),
	ASPEED_PINCTRL_FUNC(SALT10),
	ASPEED_PINCTRL_FUNC(SALT11),
	ASPEED_PINCTRL_FUNC(SALT12),
	ASPEED_PINCTRL_FUNC(SALT13),
	ASPEED_PINCTRL_FUNC(SALT14),
	ASPEED_PINCTRL_FUNC(SALT2),
	ASPEED_PINCTRL_FUNC(SALT3),
	ASPEED_PINCTRL_FUNC(SALT4),
	ASPEED_PINCTRL_FUNC(SALT5),
	ASPEED_PINCTRL_FUNC(SALT6),
	ASPEED_PINCTRL_FUNC(SALT7),
	ASPEED_PINCTRL_FUNC(SALT8),
	ASPEED_PINCTRL_FUNC(SALT9),
	ASPEED_PINCTRL_FUNC(SCL1),
	ASPEED_PINCTRL_FUNC(SCL2),
	ASPEED_PINCTRL_FUNC(SD1),
	ASPEED_PINCTRL_FUNC(SD2),
	ASPEED_PINCTRL_FUNC(SDA1),
	ASPEED_PINCTRL_FUNC(SDA2),
	ASPEED_PINCTRL_FUNC(SGPM),
	ASPEED_PINCTRL_FUNC(SGPS1),
	ASPEED_PINCTRL_FUNC(SGPS2),
	ASPEED_PINCTRL_FUNC(SIOONCTRL),
	ASPEED_PINCTRL_FUNC(SIOPBI),
	ASPEED_PINCTRL_FUNC(SIOPBO),
	ASPEED_PINCTRL_FUNC(SIOPWREQ),
	ASPEED_PINCTRL_FUNC(SIOPWRGD),
	ASPEED_PINCTRL_FUNC(SIOS3),
	ASPEED_PINCTRL_FUNC(SIOS5),
	ASPEED_PINCTRL_FUNC(SIOSCI),
	ASPEED_PINCTRL_FUNC(SPI1),
	ASPEED_PINCTRL_FUNC(SPI1CS1),
	ASPEED_PINCTRL_FUNC(SPI1DEBUG),
	ASPEED_PINCTRL_FUNC(SPI1PASSTHRU),
	ASPEED_PINCTRL_FUNC(SPI2CK),
	ASPEED_PINCTRL_FUNC(SPI2CS0),
	ASPEED_PINCTRL_FUNC(SPI2CS1),
	ASPEED_PINCTRL_FUNC(SPI2MISO),
	ASPEED_PINCTRL_FUNC(SPI2MOSI),
	ASPEED_PINCTRL_FUNC(TIMER3),
	ASPEED_PINCTRL_FUNC(TIMER4),
	ASPEED_PINCTRL_FUNC(TIMER5),
	ASPEED_PINCTRL_FUNC(TIMER6),
	ASPEED_PINCTRL_FUNC(TIMER7),
	ASPEED_PINCTRL_FUNC(TIMER8),
	ASPEED_PINCTRL_FUNC(TXD1),
	ASPEED_PINCTRL_FUNC(TXD2),
	ASPEED_PINCTRL_FUNC(TXD3),
	ASPEED_PINCTRL_FUNC(TXD4),
	ASPEED_PINCTRL_FUNC(UART6),
	ASPEED_PINCTRL_FUNC(USB11BHID),
	ASPEED_PINCTRL_FUNC(USB2AD),
	ASPEED_PINCTRL_FUNC(USB2AH),
	ASPEED_PINCTRL_FUNC(USB2BD),
	ASPEED_PINCTRL_FUNC(USB2BH),
	ASPEED_PINCTRL_FUNC(USBCKI),
	ASPEED_PINCTRL_FUNC(VGABIOSROM),
	ASPEED_PINCTRL_FUNC(VGAHS),
	ASPEED_PINCTRL_FUNC(VGAVS),
	ASPEED_PINCTRL_FUNC(VPI24),
	ASPEED_PINCTRL_FUNC(VPO),
	ASPEED_PINCTRL_FUNC(WDTRST1),
	ASPEED_PINCTRL_FUNC(WDTRST2),
};

static struct aspeed_pin_config aspeed_g5_configs[] = {
	/* GPIOA, GPIOQ */
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, B14, B13, SCU8C, 16),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   B14, B13, SCU8C, 16),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, A11, N20, SCU8C, 16),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   A11, N20, SCU8C, 16),

	/* GPIOB, GPIOR */
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, K19, H20, SCU8C, 17),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   K19, H20, SCU8C, 17),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, AA19, E10, SCU8C, 17),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   AA19, E10, SCU8C, 17),

	/* GPIOC, GPIOS*/
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, C12, B11, SCU8C, 18),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   C12, B11, SCU8C, 18),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, V20, AA20, SCU8C, 18),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   V20, AA20, SCU8C, 18),

	/* GPIOD, GPIOY */
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, F19, C21, SCU8C, 19),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   F19, C21, SCU8C, 19),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, R22, P20, SCU8C, 19),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   R22, P20, SCU8C, 19),

	/* GPIOE, GPIOZ */
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, B20, B19, SCU8C, 20),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   B20, B19, SCU8C, 20),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, Y20, W21, SCU8C, 20),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   Y20, W21, SCU8C, 20),

	/* GPIOF, GPIOAA */
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, J19, H18, SCU8C, 21),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   J19, H18, SCU8C, 21),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, Y21, P19, SCU8C, 21),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   Y21, P19, SCU8C, 21),

		/* GPIOG, GPIOAB */
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, A19, E14, SCU8C, 22),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   A19, E14, SCU8C, 22),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, N19, R20, SCU8C, 22),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   N19, R20, SCU8C, 22),

	/* GPIOH, GPIOAC */
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, A18,  D18, SCU8C, 23),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   A18,  D18, SCU8C, 23),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, G21,  G22, SCU8C, 23),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   G21,  G22, SCU8C, 23),

	/* GPIOs [I, P] */
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, C18, A15, SCU8C, 24),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   C18, A15, SCU8C, 24),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, R2,  T3,  SCU8C, 25),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   R2,  T3,  SCU8C, 25),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, L3,  R1,  SCU8C, 26),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   L3,  R1,  SCU8C, 26),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, T2,  W1,  SCU8C, 27),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   T2,  W1,  SCU8C, 27),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, Y1,  T5,  SCU8C, 28),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   Y1,  T5,  SCU8C, 28),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, V2,  T4,  SCU8C, 29),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   V2,  T4,  SCU8C, 29),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, U5,  W4,  SCU8C, 30),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   U5,  W4,  SCU8C, 30),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, V4,  V6,  SCU8C, 31),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   V4,  V6,  SCU8C, 31),

	/* GPIOs T[0-5] (RGMII1 Tx pins) */
	ASPEED_SB_PINCONF(PIN_CONFIG_DRIVE_STRENGTH, B5, B5, SCU90, 8),
	ASPEED_SB_PINCONF(PIN_CONFIG_DRIVE_STRENGTH, E9, A5, SCU90, 9),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, B5, D7, SCU90, 12),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   B5, D7, SCU90, 12),

	/* GPIOs T[6-7], U[0-3] (RGMII2 TX pins) */
	ASPEED_SB_PINCONF(PIN_CONFIG_DRIVE_STRENGTH, B2, B2, SCU90, 10),
	ASPEED_SB_PINCONF(PIN_CONFIG_DRIVE_STRENGTH, B1, B3, SCU90, 11),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, B2, D4, SCU90, 14),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   B2, D4, SCU90, 14),

	/* GPIOs U[4-7], V[0-1] (RGMII1 Rx pins) */
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, B4, C4, SCU90, 13),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   B4, C4, SCU90, 13),

	/* GPIOs V[2-7] (RGMII2 Rx pins) */
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, C2, E6, SCU90, 15),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   C2, E6, SCU90, 15),

	/* ADC pull-downs (SCUA8[19:4]) */
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, F4, F4, SCUA8, 4),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   F4, F4, SCUA8, 4),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, F5, F5, SCUA8, 5),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   F5, F5, SCUA8, 5),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, E2, E2, SCUA8, 6),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   E2, E2, SCUA8, 6),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, E1, E1, SCUA8, 7),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   E1, E1, SCUA8, 7),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, F3, F3, SCUA8, 8),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   F3, F3, SCUA8, 8),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, E3, E3, SCUA8, 9),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   E3, E3, SCUA8, 9),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, G5, G5, SCUA8, 10),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   G5, G5, SCUA8, 10),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, G4, G4, SCUA8, 11),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   G4, G4, SCUA8, 11),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, F2, F2, SCUA8, 12),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   F2, F2, SCUA8, 12),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, G3, G3, SCUA8, 13),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   G3, G3, SCUA8, 13),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, G2, G2, SCUA8, 14),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   G2, G2, SCUA8, 14),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, F1, F1, SCUA8, 15),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   F1, F1, SCUA8, 15),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, H5, H5, SCUA8, 16),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   H5, H5, SCUA8, 16),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, G1, G1, SCUA8, 17),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   G1, G1, SCUA8, 17),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, H3, H3, SCUA8, 18),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   H3, H3, SCUA8, 18),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_PULL_DOWN, H4, H4, SCUA8, 19),
	ASPEED_SB_PINCONF(PIN_CONFIG_BIAS_DISABLE,   H4, H4, SCUA8, 19),

	/*
	 * Debounce settings for GPIOs D and E passthrough mode are in
	 * SCUA8[27:20] and so are managed by pinctrl. Normal GPIO debounce for
	 * banks D and E is handled by the GPIO driver - GPIO passthrough is
	 * treated like any other non-GPIO mux function. There is a catch
	 * however, in that the debounce period is configured in the GPIO
	 * controller. Due to this tangle between GPIO and pinctrl we don't yet
	 * fully support pass-through debounce.
	 */
	ASPEED_SB_PINCONF(PIN_CONFIG_INPUT_DEBOUNCE, F19, E21, SCUA8, 20),
	ASPEED_SB_PINCONF(PIN_CONFIG_INPUT_DEBOUNCE, F20, D20, SCUA8, 21),
	ASPEED_SB_PINCONF(PIN_CONFIG_INPUT_DEBOUNCE, D21, E20, SCUA8, 22),
	ASPEED_SB_PINCONF(PIN_CONFIG_INPUT_DEBOUNCE, G18, C21, SCUA8, 23),
	ASPEED_SB_PINCONF(PIN_CONFIG_INPUT_DEBOUNCE, B20, C20, SCUA8, 24),
	ASPEED_SB_PINCONF(PIN_CONFIG_INPUT_DEBOUNCE, F18, F17, SCUA8, 25),
	ASPEED_SB_PINCONF(PIN_CONFIG_INPUT_DEBOUNCE, E18, D19, SCUA8, 26),
	ASPEED_SB_PINCONF(PIN_CONFIG_INPUT_DEBOUNCE, A20, B19, SCUA8, 27),
};

static struct regmap *aspeed_g5_acquire_regmap(struct aspeed_pinmux_data *ctx,
					       int ip)
{
	if (ip == ASPEED_IP_SCU) {
		WARN(!ctx->maps[ip], "Missing SCU syscon!");
		return ctx->maps[ip];
	}

	if (ip >= ASPEED_NR_PINMUX_IPS)
		return ERR_PTR(-EINVAL);

	if (likely(ctx->maps[ip]))
		return ctx->maps[ip];

	if (ip == ASPEED_IP_GFX) {
		struct device_node *node;
		struct regmap *map;

		node = of_parse_phandle(ctx->dev->of_node,
					"aspeed,external-nodes", 0);
		if (node) {
			map = syscon_node_to_regmap(node);
			of_node_put(node);
			if (IS_ERR(map))
				return map;
		} else
			return ERR_PTR(-ENODEV);

		ctx->maps[ASPEED_IP_GFX] = map;
		dev_dbg(ctx->dev, "Acquired GFX regmap");
		return map;
	}

	if (ip == ASPEED_IP_LPC) {
		struct device_node *np;
		struct regmap *map;

		np = of_parse_phandle(ctx->dev->of_node,
					"aspeed,external-nodes", 1);
		if (np) {
			if (!of_device_is_compatible(np->parent, "aspeed,ast2400-lpc-v2") &&
			    !of_device_is_compatible(np->parent, "aspeed,ast2500-lpc-v2") &&
			    !of_device_is_compatible(np->parent, "aspeed,ast2600-lpc-v2"))
				return ERR_PTR(-ENODEV);

			map = syscon_node_to_regmap(np->parent);
			of_node_put(np);
			if (IS_ERR(map))
				return map;
		} else
			return ERR_PTR(-ENODEV);

		ctx->maps[ASPEED_IP_LPC] = map;
		dev_dbg(ctx->dev, "Acquired LPC regmap");
		return map;
	}

	return ERR_PTR(-EINVAL);
}

static int aspeed_g5_sig_expr_eval(struct aspeed_pinmux_data *ctx,
				   const struct aspeed_sig_expr *expr,
				   bool enabled)
{
	int ret;
	int i;

	for (i = 0; i < expr->ndescs; i++) {
		const struct aspeed_sig_desc *desc = &expr->descs[i];
		struct regmap *map;

		map = aspeed_g5_acquire_regmap(ctx, desc->ip);
		if (IS_ERR(map)) {
			dev_err(ctx->dev,
				"Failed to acquire regmap for IP block %d\n",
				desc->ip);
			return PTR_ERR(map);
		}

		ret = aspeed_sig_desc_eval(desc, enabled, ctx->maps[desc->ip]);
		if (ret <= 0)
			return ret;
	}

	return 1;
}

/**
 * aspeed_g5_sig_expr_set() - Configure a pin's signal by applying an
 * expression's descriptor state for all descriptors in the expression.
 *
 * @ctx: The pinmux context
 * @expr: The expression associated with the function whose signal is to be
 *        configured
 * @enable: true to enable an function's signal through a pin's signal
 *          expression, false to disable the function's signal
 *
 * Return: 0 if the expression is configured as requested and a negative error
 * code otherwise
 */
static int aspeed_g5_sig_expr_set(struct aspeed_pinmux_data *ctx,
				  const struct aspeed_sig_expr *expr,
				  bool enable)
{
	int ret;
	int i;

	for (i = 0; i < expr->ndescs; i++) {
		const struct aspeed_sig_desc *desc = &expr->descs[i];
		u32 pattern = enable ? desc->enable : desc->disable;
		u32 val = (pattern << __ffs(desc->mask));
		struct regmap *map;

		map = aspeed_g5_acquire_regmap(ctx, desc->ip);
		if (IS_ERR(map)) {
			dev_err(ctx->dev,
				"Failed to acquire regmap for IP block %d\n",
				desc->ip);
			return PTR_ERR(map);
		}

		/*
		 * Strap registers are configured in hardware or by early-boot
		 * firmware. Treat them as read-only despite that we can write
		 * them. This may mean that certain functions cannot be
		 * deconfigured and is the reason we re-evaluate after writing
		 * all descriptor bits.
		 *
		 * Port D and port E GPIO loopback modes are the only exception
		 * as those are commonly used with front-panel buttons to allow
		 * normal operation of the host when the BMC is powered off or
		 * fails to boot. Once the BMC has booted, the loopback mode
		 * must be disabled for the BMC to control host power-on and
		 * reset.
		 */
		if (desc->ip == ASPEED_IP_SCU && desc->reg == HW_STRAP1 &&
		    !(desc->mask & (BIT(21) | BIT(22))))
			continue;

		if (desc->ip == ASPEED_IP_SCU && desc->reg == HW_STRAP2)
			continue;

		/* On AST2500, Set bits in SCU70 are cleared from SCU7C */
		if (desc->ip == ASPEED_IP_SCU && desc->reg == HW_STRAP1) {
			u32 value = ~val & desc->mask;

			if (value) {
				ret = regmap_write(ctx->maps[desc->ip],
						   HW_REVISION_ID, value);
				if (ret < 0)
					return ret;
			}
		}

		ret = regmap_update_bits(ctx->maps[desc->ip], desc->reg,
					 desc->mask, val);

		if (ret)
			return ret;
	}

	ret = aspeed_sig_expr_eval(ctx, expr, enable);
	if (ret < 0)
		return ret;

	if (!ret)
		return -EPERM;

	return 0;
}

static const struct aspeed_pin_config_map aspeed_g5_pin_config_map[] = {
	{ PIN_CONFIG_BIAS_PULL_DOWN,  0, 1, BIT_MASK(0)},
	{ PIN_CONFIG_BIAS_PULL_DOWN, -1, 0, BIT_MASK(0)},
	{ PIN_CONFIG_BIAS_DISABLE,   -1, 1, BIT_MASK(0)},
	{ PIN_CONFIG_DRIVE_STRENGTH,  8, 0, BIT_MASK(0)},
	{ PIN_CONFIG_DRIVE_STRENGTH, 16, 1, BIT_MASK(0)},
};

static const struct aspeed_pinmux_ops aspeed_g5_ops = {
	.eval = aspeed_g5_sig_expr_eval,
	.set = aspeed_g5_sig_expr_set,
};

static struct aspeed_pinctrl_data aspeed_g5_pinctrl_data = {
	.pins = aspeed_g5_pins,
	.npins = ARRAY_SIZE(aspeed_g5_pins),
	.pinmux = {
		.ops = &aspeed_g5_ops,
		.groups = aspeed_g5_groups,
		.ngroups = ARRAY_SIZE(aspeed_g5_groups),
		.functions = aspeed_g5_functions,
		.nfunctions = ARRAY_SIZE(aspeed_g5_functions),
	},
	.configs = aspeed_g5_configs,
	.nconfigs = ARRAY_SIZE(aspeed_g5_configs),
	.confmaps = aspeed_g5_pin_config_map,
	.nconfmaps = ARRAY_SIZE(aspeed_g5_pin_config_map),
};

static const struct pinmux_ops aspeed_g5_pinmux_ops = {
	.get_functions_count = aspeed_pinmux_get_fn_count,
	.get_function_name = aspeed_pinmux_get_fn_name,
	.get_function_groups = aspeed_pinmux_get_fn_groups,
	.set_mux = aspeed_pinmux_set_mux,
	.gpio_request_enable = aspeed_gpio_request_enable,
	.strict = true,
};

static const struct pinctrl_ops aspeed_g5_pinctrl_ops = {
	.get_groups_count = aspeed_pinctrl_get_groups_count,
	.get_group_name = aspeed_pinctrl_get_group_name,
	.get_group_pins = aspeed_pinctrl_get_group_pins,
	.pin_dbg_show = aspeed_pinctrl_pin_dbg_show,
	.dt_node_to_map = pinconf_generic_dt_node_to_map_all,
	.dt_free_map = pinctrl_utils_free_map,
};

static const struct pinconf_ops aspeed_g5_conf_ops = {
	.is_generic = true,
	.pin_config_get = aspeed_pin_config_get,
	.pin_config_set = aspeed_pin_config_set,
	.pin_config_group_get = aspeed_pin_config_group_get,
	.pin_config_group_set = aspeed_pin_config_group_set,
};

static struct pinctrl_desc aspeed_g5_pinctrl_desc = {
	.name = "aspeed-g5-pinctrl",
	.pins = aspeed_g5_pins,
	.npins = ARRAY_SIZE(aspeed_g5_pins),
	.pctlops = &aspeed_g5_pinctrl_ops,
	.pmxops = &aspeed_g5_pinmux_ops,
	.confops = &aspeed_g5_conf_ops,
};

static int aspeed_g5_pinctrl_probe(struct platform_device *pdev)
{
	int i;

	for (i = 0; i < ARRAY_SIZE(aspeed_g5_pins); i++)
		aspeed_g5_pins[i].number = i;

	aspeed_g5_pinctrl_data.pinmux.dev = &pdev->dev;

	return aspeed_pinctrl_probe(pdev, &aspeed_g5_pinctrl_desc,
			&aspeed_g5_pinctrl_data);
}

static const struct of_device_id aspeed_g5_pinctrl_of_match[] = {
	{ .compatible = "aspeed,ast2500-pinctrl", },
	/*
	 * The aspeed,g5-pinctrl compatible has been removed the from the
	 * bindings, but keep the match in case of old devicetrees.
	 */
	{ .compatible = "aspeed,g5-pinctrl", },
	{ },
};

static struct platform_driver aspeed_g5_pinctrl_driver = {
	.probe = aspeed_g5_pinctrl_probe,
	.driver = {
		.name = "aspeed-g5-pinctrl",
		.of_match_table = aspeed_g5_pinctrl_of_match,
	},
};

static int aspeed_g5_pinctrl_init(void)
{
	return platform_driver_register(&aspeed_g5_pinctrl_driver);
}

arch_initcall(aspeed_g5_pinctrl_init);
