#-*- coding: utf-8 -*-

# Copyright 2008-2013 Calculate Ltd. http://www.calculate-linux.org
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.

import os
import sys
import re
from os import path
from calculate.lib.datavars import (Variable, VariableError,
    ReadonlyVariable, ReadonlyTableVariable, TableVariable, FieldValue,
    SimpleDataVars, DataVarsError)
from calculate.lib.utils.portage import searchProfile
from calculate.lib.utils.files import readLinesFile, readFile, makeDirectory, \
    listDirectory

from calculate.lib.cl_lang import setLocalTranslate
from calculate.lib.utils.text import simplify_profiles
from calculate.update.package_tools import Git, GitError, Layman
from calculate.update.profile import RepositoryStorageSet, DEFAULT_BRANCH

from calculate.lib.variables.linux import VariableOsLinuxName, \
    VariableOsLinuxSubname, VariableOsLinuxVer, VariableOsLinuxShortname


setLocalTranslate('cl_update3', sys.modules[__name__])

class VariableAcUpdateSync(ReadonlyVariable):
    """
    Action variable which has value "up" for package install and
    install this package
    """
    def get(self):
        action = self.Get("cl_action")
        if action in ("sync",):
            if self.Get('cl_update_world'):
                return self.Get('cl_update_world')
        return ""

class VariableClUpdateWorld(Variable):
    def get(self):
        if self.Get('cl_rebuild_world_set') == "on":
            return "rebuild"
        else:
            return ""

class VariableClRebuildWorldSet(Variable):
    """
    List of action update world, rebuild world, 
    """
    type = "bool"
    opt = ["--rebuild-world"]
    untrusted = True
    value = "off"

    def init(self):
        self.help = _("Rebuild world")
        self.label = _("Rebuild world")

class VariableClUpdateRevSet(Variable):
    """
    List of action update world, rebuild world, 
    """
    type = "bool"
    opt = ["--update-rev"]
    untrusted = True
    value = "on"
    check_after = ["cl_update_sync_rep",
                    "cl_update_metadata_force",
                    "cl_update_other_set",
                    "cl_update_eixupdate_force"]

    def init(self):
        self.help = _("make a revision update")
        self.label = _("Make a revision update")

class VariableClUpdateRep(Variable):
    """
    Обновлять репозитории до конкретной ревизии или до последней
    """
    type = "choice"
    value = "rev"

    def choice(self):
        return ["last","rev"]

class VariableClUpdateRepData(ReadonlyTableVariable):
    """
    Информация о репозиториях
    """
    source = ['cl_update_rep_name',
              'cl_update_rep_url',
              'cl_update_rep_path',
              'cl_update_rep_rev',
              'cl_update_branch_name']

class VariableClUpdateRepName(Variable):
    """
    Список имен используемых репозиториев
    """
    type = "list"
    value = []

class VariableClUpdateRepUrl(Variable):
    """
    Список путей до репозиториев
    """
    type = "list"
    value = []

class VariableClUpdateLaymanStorage(ReadonlyVariable):
    """
    Путь к репозиториям layman
    """
    def get(self):
        laymanConf = "/etc/layman/layman.cfg"
        reStorage = re.compile("^storage\s*:\s*(\S+)")
        if path.exists(laymanConf):
            for line in readLinesFile(laymanConf):
                match = reStorage.search(line)
                if match:
                    return match.group(1)
        return "/var/lib/layman"

class VariableClUpdateRepPath(ReadonlyVariable):
    """
    Пути до репозиториев
    """
    type = "list"
    mapPath = {'portage':'/usr/portage'}

    def get(self):
        repPath = self.Get('cl_update_layman_storage')
        def generatePaths(names):
            for name in names:
                if name in self.mapPath:
                    yield self.mapPath[name]
                else:
                    yield path.join(repPath,name)
        return list(generatePaths(self.Get('cl_update_rep_name')))

class VariableClUpdateRepRev(Variable):
    """
    Ревизии до которых необходимо обновить репозитории
    """
    type = "list"

    def get(self):
        if self.Get('cl_update_rep') == 'rev':
            revPaths = searchProfile(self.Get('cl_profile_system'),
                                     "rev")
            if revPaths:
                revPath = revPaths[-1]
                dictNamesRevs = dict(map(lambda x:x.strip().partition('=')[::2],
                                readLinesFile(revPath)))
                return map(lambda x:dictNamesRevs.get(x,"last"),
                       self.Get('cl_update_rep_name'))
        return ["last"]*len(self.Get('cl_update_rep_name'))

class VariableClUpdateBranch(TableVariable):
    """
    Выбор веток репозиториев до которых необходимо обновиться
    """
    opt = ["--branch"]
    metavalue = 'BRANCHES'
    untrusted = True
    source = ["cl_update_branch_rep",
              "cl_update_branch_name"]

    def init(self):
        self.help = _("set branches for repository (REPOSITORY:BRANCH)")
        self.label = _("Repositories branches")

    def raiseReadonlyIndexError(self,fieldname="",variablename="",
                                value=""):
        """
        Неизвестный оврелей
        """
        raise VariableError(_("Repository %s not found")%value)

class VariableClUpdateBranchRep(ReadonlyVariable):
    """
    Список доступных репозиторием
    """
    type = "list"

    def init(self):
        self.label = _("Repositories")

    def get(self):
        return self.Get('cl_update_rep_name')


class VariableClUpdateBranchName(Variable):
    """
    Список доступных репозиторием
    """
    type = "choiceedit-list"

    def init(self):
        self.label = _("Branches")

    def choice(self):
        return ["master", "develop", "update"]

    def get(self):
        def generateBranch():
            for reppath in self.Get('cl_update_rep_path'):
                headPath = path.join(reppath, ".git/HEAD")
                yield readFile(headPath).rpartition('/')[2].strip() or "master"

        return list(generateBranch())


class VariableClUpdateSyncRep(Variable):
    """
    Обновляемый репозиторий
    """
    type = "choice-list"
    element = "selecttable"
    opt = ["cl_update_sync_rep"]
    metavalue = "REPOSITORIES"
    untrusted = True

    def init(self):
        self.help = _("synchronized repositories (all by default)")
        self.label = _("Synchronized repositories")

    def set(self,value):
        orderList = self.Get('cl_update_rep_name')
        return sorted(value,key=lambda x:
            (orderList.index(x) if x in orderList else -1),reverse=True)
    
    def get(self):
        return list(reversed(self.Get('cl_update_rep_name')))

    def choice(self):
        return self.Get('cl_update_rep_name')

class VariableClUpdateSyncOverlayRep(ReadonlyVariable):
    """
    Обновляемые репозитории (исключая portage)
    """
    type = "list"

    def get(self):
        return filter(lambda x:x!="portage",self.Get('cl_update_sync_rep'))

class VariableClUpdateOutdateSet(ReadonlyVariable):
    """
    Флаг устанавливаемый в ходе обновления репозиториев,
    сообщающий что хотя бы один из запланированных репозиториев
    обновлен и следует обновляет различные метаданные

    Если обновляются прочие оверлеи - данные считаются что устарели
    """
    type = "bool"
    value = "off"

    def get(self):
        return self.Get('cl_update_other_set')

class VariableClUpdateMetadataForce(Variable):
    """
    Принудительное действие с метаданными
    """
    type = "choice"
    value = "auto"
    opt = ["--update-metadata"]
    syntax = "--{choice}-update-metadata"
    metavalue = "MODE"
    #untrusted = True

    def init(self):
        self.help = ("'force' - " + _("force the update ebuilds metadata") + 
                     ",\n'skip' - " + _("skip the ebuild metadata update") +
                     ",\n'auto' - " + _("update metadata if it is outdated"))
        self.label = _("Update metadata")

    def choice(self):
        return [("force", _("Force")),
                ("skip", _("Skip")),
                ("auto", _("If needed"))]

class VariableClUpdateEixupdateForce(Variable):
    """
    Принудительное действие с eix-update
    """
    type = "choice"
    value = "auto"
    opt = ["--eix-update"]
    syntax = "--{choice}-eix-update"
    metavalue = "MODE"
    #untrusted = True

    def init(self):
        self.help = ("'force' - " + _("force the eix cache update") +
                     ",\n'skip' - " + _("skip the eix cache update") +
                     ",\n'auto' - " + _("update the eix cache if it "
                                        "is outdated"))
        self.label = _("Update the eix cache")

    def choice(self):
        return [("force", _("Force")),
                ("skip", _("Skip")),
                ("auto", _("If needed"))]

class VariableClUpdateOtherSet(Variable):
    """
    Обновить остальные оверлеи
    """
    type = "bool"
    value = "off"
    opt = ["-o", "--update-other"]

    def init(self):
        self.help = _("update other overlays")
        self.label = _("Update other overlays")

class VariableClUpdateOtherRepData(ReadonlyTableVariable):
    """
    Информация о прочих репозиториях
    """
    source = ['cl_update_other_rep_name',
              'cl_update_other_rep_path']

    def generator(self):
        repNames = self.Get('cl_update_rep_name')
        layman = Layman(self.Get('cl_update_layman_installed'),
                        self.Get('cl_update_layman_make'))
        layman_overlays = layman.get_installed()
        for rpath in self.Get('cl_portdir_overlay'):
            repo_file = path.join(rpath, "profiles/repo_name")
            rname = readFile(repo_file).strip() or path.basename(rpath)
            if rname in layman_overlays and not rname in repNames:
                yield (rname, rpath)

    def get(self):
        return list(self.generator())

class VariableClUpdateOtherRepName(FieldValue,ReadonlyVariable):
    """
    Список имен прочих репозиториев
    """
    type = "list"
    source_variable = "cl_update_other_rep_data"
    column = 0

class VariableClUpdateOtherRepPath(FieldValue,ReadonlyVariable):
    """
    Список путей до прочих репозиториев
    """
    type = "list"
    source_variable = "cl_update_other_rep_data"
    column = 1

class VariableClUpdateLaymanInstalled(Variable):
    """
    Путь до файла layman installed.xml
    """
    # TODO: извлечь из layman.cfg
    value = "/var/lib/layman/installed.xml"

class VariableClUpdateLaymanMake(Variable):
    """
    Путь до файла make.conf изменяемого layman`ом
    """
    # TODO: извлечь из layman.cfg
    value = "/var/lib/layman/make.conf"

class VariableClUpdatePretendSet(Variable):
    """
    Запустить предварительную проверку на обновления
    """
    type = "bool"
    value = "off"
    opt = ["-p", "--pretend"]

    def init(self):
        self.label = _("Pretend a package update")
        self.help = _("instead of actually performing the update, "
                      "simply display the list of packages that "
                      "will be installed")

class VariableClUpdateSyncOnlySet(Variable):
    """
    Не выполнять установку/обновление пакетов при обновлении
    """
    type = "bool"
    value = "off"
    opt = ["-s", "--sync"]

    def init(self):
        self.label = _("Only synchronize repositories")
        self.help = _("do not update packages")

    def check(self, value):
        if (value == "on" and self.Get('cl_rebuild_world_set') != 'on' and
                not self.Get('cl_update_sync_rep') and
                    self.Get('cl_update_other_set') == 'off' and
                    self.Get('cl_update_rev_set') == 'off' and
                    self.Get('cl_update_metadata_force') != "force" and
                    self.Get('cl_update_eixupdate_force') != "force"):
            raise VariableError(_("Select at least one sync repository"))

class VariableClUpdateWaitAnotherSet(Variable):
    """
    Ждать завершения другого процесса обновления
    """
    type = "bool"
    value = "on"
    opt = ["--wait-another-update"]

    def init(self):
        self.label = _("Wait for another update to be complete")
        self.help = _("wait until the running update is finished")

class VariableClUpdateProfileStorage(ReadonlyVariable):
    type = "object"

    def get(self):
        return RepositoryStorageSet()

class VariableClUpdateRepHost(Variable):
    type = "list"
    value = ['calculate']

class VariableClUpdateRepHostUrl(Variable):
    type = "list"
    value = ['git://git.calculate.ru/calculate/%s.git']


class VariableClUpdateProfileDatavars(ReadonlyVariable):
    type = "object"

    def get(self):
        profile = self.Get('cl_update_profile_system')
        path_profile = self.Select('cl_profile_path',
                                   where='cl_profile_shortname',
                                   eq=profile, limit=1)
        if path_profile:
            return DataVarsUpdateProfile(path_profile)
        return ""

class VariableClUpdateProfileLinuxFullname(ReadonlyVariable):
    """
    Имя системы в профиле
    """
    def init(self):
        self.label = _("Distribution name")

    def get(self):
        dv = self.Get('cl_update_profile_datavars')
        if dv:
            try:
                subname = dv.Get('os_linux_subname')
                linuxname = dv.Get('os_linux_name')
                linuxver = dv.Get('os_linux_ver')
                if subname:
                    return "%s %s %s" % (linuxname, linuxver, subname)
                return "%s %s" %(linuxname,linuxver)
            except DataVarsError:
                raise VariableError("Wrong Calculate Linux profile")
        return ""

class VariableClUpdateProfileDependData(ReadonlyTableVariable):
    """
    Зависимые репозитории
    """
    source = ['cl_update_profile_depend_name',
              'cl_update_profile_depend_url']

    def init(self):
        self.label = _("Used repositories")

    @staticmethod
    def url_like(url1, url2):
        match1 = VariableClUpdateProfileRep.re_url.search(url1)
        match2 = VariableClUpdateProfileRep.re_url.search(url2)
        if match1 and match2:
            return match1.group(2).lower() == match2.group(2).lower()
        return False


    def get(self, hr=False):
        dv = self.Get('cl_update_profile_datavars')
        url = self.Get('cl_update_profile_rep').lower()
        if dv:
            return reversed(zip(dv.Get('cl_update_rep_name'),
                                dv.Get('cl_update_rep_url')))
        return ""

    setValue = Variable.setValue

class VariableClUpdateProfileDependName(FieldValue, ReadonlyVariable):
    type = "list"
    source_variable = "cl_update_profile_depend_data"
    column = 0

    def init(self):
        self.label = _("Name")

class VariableClUpdateProfileDependUrl(FieldValue, ReadonlyVariable):
    type = "list"
    source_variable = "cl_update_profile_depend_data"
    column = 1

    def init(self):
        self.label = _("URL")


class VariableClUpdateProfileRepName(ReadonlyVariable):
    type = "list"

    def get(self):
        dv = self.Get('cl_update_profile_datavars')
        if dv:
            return dv.Get('cl_update_rep_name')
        return []

class VariableClUpdateProfileSyncRep(ReadonlyVariable):
    type = "list"

    def get(self):
        return list(reversed(self.Get('cl_update_profile_rep_name')))

class VariableClUpdateProfileRepUrl(ReadonlyVariable):
    type = "list"

    def get(self):
        dv = self.Get('cl_update_profile_datavars')
        if dv:
            return dv.Get('cl_update_rep_url')
        return []

class VariableClUpdateProfileLinuxVer(ReadonlyVariable):
    """
    Имя системы в профиле
    """
    def init(self):
        self.label = _("System profile version")

    def get(self):
        dv = self.Get('cl_update_profile_datavars')
        if dv:
            return dv.Get('os_linux_ver')
        return ""

class VariableClUpdateProfileLinuxName(ReadonlyVariable):
    """
    Имя системы в профиле
    """
    def get(self):
        dv = self.Get('cl_update_profile_datavars')
        if dv:
            dv.Get('os_linux_name')
        return ""

class VariableClUpdateProfileRep(Variable):
    """
    Текущий репозиторий
    """
    untrusted = True
    check_after = "cl_update_profile_branch"

    opt = ["--url"]

    def init(self):
        self.label = _("Profile repository")
        self.help = _("set the profile repository")

    re_url = re.compile(
        r"^(?:(%s)://)?(\w[\w\./:-]+?\w)(\.git)?$" % "|".join(
            ["http", "https", "git"]))

    re_shortname = re.compile('^(?:([\w\.-]+):)?([\w\.-]+)$')

    @classmethod
    def normalize_url(cls, url):
        match = cls.re_url.match(url)
        if not match:
            raise VariableError(_("Wrong repository URL"))
        url = match.group(2)
        url = "%s://%s" % (match.group(1) or "git", url)
        url = "%s.git" % url
        return url

    def url_by_shortname(self, value):
        match = self.re_shortname.match(value)
        if not match.group(1):
            template = self.Get('cl_update_rep_host_url')
            if template:
                template = template[0]
        else:
            template = self.Select('cl_update_rep_host_url',
                                   where='cl_update_rep_host',
                                   eq=match.group(1), limit=1)
        if not template:
            raise VariableError(_("Failed to determine the repository host"))

        try:
            return template % match.group(2)
        except TypeError:
            raise VariableError(_("Failed to determine the repository host"))

    def set(self, value):
        if self.re_shortname.match(value):
            value = self.url_by_shortname(value)
        return self.normalize_url(value)

    def check(self, value):
        if not value:
            raise VariableError("Need to specify profile repository")
        try:
            branch = self.Get('cl_update_profile_branch')
            self.Get('cl_update_profile_storage').get_profiles(value, branch)
        except GitError as e:
            raise VariableError(str(e))
        if not self.Get('cl_profile_shortname'):
            raise VariableError(_("Repository %s has no profiles")%value)

    def get(self):
        try:
            profile = self.Get('cl_profile_system')
            while profile != '/' and ".git" not in listDirectory(profile):
                profile = path.dirname(profile)
            if profile == '/':
                return ""
            git = Git()
            return git.get_url(profile, "origin") or ""
        except:
            return "git://git.calculate.ru/calculate/distros.git"

class VariableClUpdateProfileRepoName(ReadonlyVariable):
    def init(self):
        self.label = _("Repository name")

    def get(self):
        rep_set = self.Get('cl_update_profile_storage')
        url = self.Get('cl_update_profile_rep')
        rep = rep_set.get_repository(url)
        if rep:
            return rep.repo_name
        return ""


class VariableClUpdateProfileBranch(Variable):
    """
    Текущий репозиторий
    """
    untrusted = True

    type = "edit"
    opt = ["--branch"]

    def init(self):
        self.label = _("Repository branch")
        self.help = _("set the repository branch")

    def check(self, value):
        pass
        ## TODO: проверка ветки
        #try:
        #    url = self.Get('cl_update_profile_rep')
        #    self.Get('cl_update_profile_storage').get_profiles(url, value)
        #except GitError as e:
#            raise VariableError(str(e))

    def get(self):
        rep_set = self.Get('cl_update_profile_storage')
        url = self.Get('cl_update_profile_rep')
        #print url, rep_set.is_local(url, branch=None)
        if rep_set.is_local(url, branch=None):
            rep = rep_set.get_repository(url, branch=None)
            git = Git()
            return git.getBranch(rep.directory)
        return DEFAULT_BRANCH


class VariableClProfileSystem(ReadonlyVariable):
    """
    Профиль системы (симлинк /etc/make.profile')
    """
    def get(self):
        try:
            make_profile = self.Get('cl_make_profile')
            return path.normpath(
                path.join(path.dirname(make_profile),
                          os.readlink(make_profile)))
        except:
            raise VariableError(_("Failed to determine the system profile"))


class VariableClProfileData(ReadonlyTableVariable):
    source = ["cl_profile_fullname",
              "cl_profile_shortname",
              "cl_profile_path",
              "cl_profile_arch"]

    def get(self, hr=False):
        url = self.Get('cl_update_profile_rep')
        if not url:
            return [[]]
        try:
            rep_set = self.Get('cl_update_profile_storage')
            branch = self.Get('cl_update_profile_branch')
            rep = rep_set.get_repository(url, branch)
            if rep and self.Get('cl_update_profile_sync_set') == 'on':
                rep.sync()
            profiles = rep_set.get_profiles(url, branch)
        except GitError:
            return [[]]
        arch = self.Get('os_arch_machine_gentoo')
        if profiles:
            repo_name = profiles[0].repository.repo_name
            arch_profiles = [x for x in profiles if x.arch == arch]
            full_name = [x.profile for x in arch_profiles]
            profile_path = [x.path for x in arch_profiles]
            profile_arch = [x.arch for x in arch_profiles]
            short_name = simplify_profiles(full_name)
            full_name = ["%s:%s"%(repo_name,x) for x in full_name]
            return zip(full_name,
                       short_name,
                       profile_path,
                       profile_arch)
        return [[]]

    setValue = Variable.setValue


class VariableClProfileFullname(FieldValue, ReadonlyVariable):
    """
    Полное название профиля
    """
    type = "list"
    source_variable = "cl_profile_data"
    column = 0

class VariableClProfileShortname(FieldValue, ReadonlyVariable):
    """
    Упрощенное название профиля
    """
    type = "list"
    source_variable = "cl_profile_data"
    column = 1

class VariableClProfilePath(FieldValue, ReadonlyVariable):
    """
    Путь от корня до профиля
    """
    type = "list"
    source_variable = "cl_profile_data"
    column = 2

class VariableClProfileArch(FieldValue, ReadonlyVariable):
    """
    Архитектура профиля
    """
    type = "list"
    source_variable = "cl_profile_data"
    column = 3

class VariableClUpdateProfileSystem(Variable):
    """
    Профиль системы (симлинк /etc/make.profile')
    """
    type = "choice"
    opt = ["cl_update_profile_system"]
    untrusted = True
    metavalue = "PROFILE"

    def init(self):
        self.label = _("System profile")
        self.help = _("set the system profile")

    def check(self, profile):
        if not profile:
            raise VariableError(_("You must specify the profile"))
        path_profile = self.Select('cl_profile_path',
                                   where='cl_profile_shortname',
                                   eq=profile, limit=1)
        if path_profile:
            dv = DataVarsUpdateProfile(path_profile)
            try:
                if (not dv.Get('cl_update_rep_name') or
                    not dv.Get('cl_update_rep_url')):
                    raise VariableError(_("Repository variables "
                        "were not configured for the profile"))
                if not dv.Get('os_linux_name'):
                    raise VariableError()
            except (DataVarsError, VariableError) as e:
                raise VariableError(_("Profile is not Calculate"))
        else:
            raise VariableError(_("Wrong Calculate profile"))

    def get(self):
        try:
            profile_system = self.Get('cl_profile_system')
            profile = self.Select('cl_profile_shortname',
                                  where='cl_profile_path',
                                  eq=profile_system, limit=1)
            if profile:
                return profile
        except VariableError:
            pass
        shortname = self.Get('cl_profile_shortname')
        if len(shortname) == 1:
            return shortname[0]
        return ""

    def choice(self):
        url = self.Get('cl_update_profile_rep')
        if not url:
            return []
        arch = self.Get('os_arch_machine_gentoo')
        profiles = zip(*self.Select(['cl_profile_shortname',
                                     'cl_profile_fullname'],
                                    where='cl_profile_arch', eq=arch))
        if profiles:
            short_name, full_name = profiles
            return zip(short_name, full_name)
        return []

class DataVarsUpdateProfile(SimpleDataVars):
    """
    Упрощенная модель переменных для получения данных с удаленного профиля
    """
    source = ['cl_update_rep_name',
              'cl_update_rep_url',
              'cl_update_rep_path',
              'cl_update_rep_rev',
              'cl_update_branch_name']
    def __init__(self, profile):
        SimpleDataVars.__init__(self,
                                VariableOsLinuxName(),
                                VariableOsLinuxShortname(),
                                VariableOsLinuxSubname(),
                                VariableOsLinuxVer(),
                                VariableClUpdateRepData(section="update"),
                                VariableClUpdateRepPath(section="update"),
                                VariableClUpdateRepRev(section="update"),
                                VariableClUpdateBranchName(section="update"),
                                VariableClProfileSystem(section="update"),
                                VariableClUpdateLaymanStorage(section="update"),
                                VariableClUpdateRepName(section="update"),
                                VariableClUpdateRep(section="update"),
                                VariableClUpdateRepUrl(section="update"))
        self.cache['cl_profile_system'] = profile
        self.flIniFileFrom(profile)

    def __repr__(self):
        return "Profile variables"

class VariableClUpdateProfileSyncSet(Variable):
    """
    Не выполнять установку/обновление пакетов при обновлении
    """
    type = "bool"
    value = "off"
    opt = ["-s", "--sync"]

    def init(self):
        self.label = _("Synchronize repositories")
        self.help = _("synchronize repositories")
